<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

use \Michelf\Markdown;

class Admin extends MY_Controller
{
	private $sites;
	private $data = array("show_menu" => true);

	public function __construct()
	{
		parent::__construct();

		if(INSTALLED)
		{
			$this->load->model("page_tag");
			$this->current_user = get_user();
			$this->data["user"] = $this->current_user;
			$this->data["tags"] = $this->page_tag->get();
		}
		else
		{
			redirect("/install", "location");
			return false;
		}
	}

	protected function init()
	{
		$this->admin_login_required();
		$this->load->model(array("site", "moduleinstance"));

		if(!Mainframe::adminInit())
		{
			redirect("/install", "location");
			return false;
		}

		$site = Mainframe::site();
		$this->sites = $this->site->get();
		$this->data["sites"] = $this->sites;

		$key = "module_js:" . Mainframe::active_site_id() . ":" . $this->current_user->user_id;

		if(($module_js = $this->cache->get($key)) === false)
		{
			$modules = $this->moduleinstance->LoadBySiteID(Mainframe::active_site_id(), "tag", $this->current_user);
			$module_js = "[";

			foreach($modules as $module)
			{
				$module_js .= "[\"$module->tag ($module->name)\", $module->module_instance_id], ";
			}
			$module_js = substr($module_js, 0, strlen($module_js)-2);
			$module_js .= "]";

			if($module_js == "]")
			{
				$module_js = "null";
			}

			$this->cache->save($key, $module_js, 86400);
		}

		Mainframe::setModuleJS($module_js);

		return true;
	}

	public function index()
	{
		$this->init();

		$this->load->model("template");

		$site = Mainframe::site();
		$template = new Template();
		$template->Load($site->template_id);
		Mainframe::setTemplate($template);

		$user = Mainframe::user();

		if(!$template->template_id || !file_exists(TEMPLATE_PATH . "/$template->directory/index.php"))
		{
			FeralRedirect("/admin/templates");
		}

		$this->data["matomo_site_id"]    = $site->matomo_site_id;
		$this->data["matomo_token_auth"] = $site->matomo_token_auth;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/dashboard", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function media()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MEDIA))
		{
			return $this->denied($this->data);
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/media", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	/* generic functions
	 * Most list and delete functions are identical except for the class and ID used.
	 * The "generic" functions will be called from the specific functions when possible to save tons of duplicate code.
	*/
	private function generic_list($model, $dataset)
	{
		$this->init();

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/" . $model . "_list", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	private function generic_delete($model, $models, $id="")
	{
		$this->init();

		$this->load->model("$model");

		if(!$id)
		{
			$id = get_id();// $this->input->get($model . "_id");
		}

		if($id)
		{
			//get the class name so we know what object to create
			$class = ucwords($model);
			$obj = new $class();
			$obj->load($id);

			//format the class name pretty for display
			$class = ucwords(str_replace("_", " ", $class));

			if($obj->delete())
			{
				$this->cache->clean();
				$this->messages[] = "$class Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting $class<br /><br />You can't delete a $class that has data linked to it.";
			}
		}

		$this->$models();
	}
	/* end generic functions */

	/* user functions */
	public function users()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_USERS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("user", "users");
	}

	public function user($id=0, $edit_user=null)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_USERS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("shop/shop_shopper_group");

		if(!$edit_user)
		{
			$edit_user = new User();
			$edit_user->load($id);
		}

		$shopper_groups = $this->shop_shopper_group->LoadBySiteID(Mainframe::active_site_id());
		$roles 			= $this->acl_role->loadByUserID($edit_user->user_id);

		foreach($roles as $role)
		{
			$role->actions = $this->acl_action->loadByRoleID($role->acl_role_id);
		}

		$this->data["id"] 				= $edit_user->user_id;
		$this->data["edit_user"] 		= $edit_user;
		$this->data["shopper_groups"] 	= $shopper_groups;
		$this->data["roles"]			= $roles;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/user_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function user_save()
	{
		$this->init();
		csrf_verify();
		$this->load->model("acl_user_role");

		if(!$this->current_user->has(ACTION_USERS))
		{
			return $this->denied($this->data);
		}

		$id = $this->input->post("user_id");

		$check_user = new User();
		$edit_user = new User();
		$edit_user->load($id);
		$edit_user->readPostVars();

		if($this->input->post("password") && $this->input->post("password") != $this->input->post("password2"))
		{
			$this->errors[] = "Passwords don't match.";
			return $this->user($id, $edit_user);
		}
		else if($check_user->LoadByUsername($this->input->post("username")) && $check_user->user_id != $edit_user->user_id)
		{
			$this->errors[] = "Username already exists!";
			return $this->user($id, $edit_user);
		}
		else if($check_user->LoadByEmail($this->input->post("email")) && $check_user->user_id != $edit_user->user_id)
		{
			$this->errors[] = "E-mail address already exists!";
			return $this->user($id, $edit_user);
		}

		$saved = $edit_user->save();

		$roles = $this->acl_role->get();

		foreach($roles as $role)
		{
			$checked = $this->input->post("role_" . $role->acl_role_id);

			if($checked)
			{
				$this->acl_user_role->add_link($edit_user->user_id, $role->acl_role_id);
			}
			else
			{
				$this->acl_user_role->delete_link($edit_user->user_id, $role->acl_role_id);
			}
		}

		// if(!$id)
		// {
		// 	//new record, continue editing so we can add roles
		// 	$message = "<p>An account has been created for you on " . COMPANY_NAME . ".</p>";
		// 	$message .= "<p><a href='" . LIVE_SITE . "' target='_blank'>" . LIVE_SITE . "</a></p>";
		// 	$message .= "<p>Your credentials are:<br /><br /><b>Username:</b> " . $edit_user->username . "<br /><b>Password:</b> " . $this->input->post("password") . "</p>";

		// 	send_notification($edit_user, "New User Account", $message);
		// }
		// else
		// {
		// 	if($this->input->post("password"))
		// 	{
		// 		$message = "<p>Your account on " . COMPANY_NAME . " has been updated.</p>";
		// 		$message .= "<p><a href='" . LIVE_SITE . "' target='_blank'>" . LIVE_SITE . "</a></p>";
		// 		$message .= "<p>Your new credentials are:<br /><br /><b>Username:</b> " . $edit_user->username . "<br /><b>Password:</b> " . $this->input->post("password") . "</p>";

		// 		send_notification($edit_user, "Account Updated", $message);
		// 	}
		// }

		if($saved)
		{
			$this->cache->clean();
			$this->messages[] = "User Saved";
		}
		else
		{
			$this->errors[] = "Error Saving User";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->user($edit_user->user_id);
		}
		else
		{
			$this->users();
		}
	}

	public function user_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_USERS))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("user", "users");
	}
	/* end user functions */

	/* domain functions */
	public function domains()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("domain", "domains");
	}

	public function domain($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("domain", "site"));

		$domain = new Domain();
		$domain->load($id);

		$sites = $this->site->get();

		$this->data["id"] = $id;
		$this->data["domain"] = $domain;
		$this->data["sites"] = $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/domain_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function domain_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model("domain");
		$id = $this->input->post("domain_id");

		$domain = new Domain();
		$domain->load($id);
		$domain->readPostVars();

		if($domain->save())
		{
			$this->cache->clean();
			$this->messages[] = "Domain Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Domain";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->domain($domain->domain_id);
		}
		else
		{
			$this->domains();
		}
	}

	public function domain_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("domain", "domains");
	}
	/* end domain functions */

	/* site functions */
	public function sites()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("site", "sites");
	}

	public function site($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("template", "site", "page", "domain"));
		$this->load->library("encryption");

		$site = new Site();
		$site->load($id);
		$site->mail_password = $this->encryption->decrypt($site->mail_password);

		$domains 		= $this->domain->get();
		$templates 		= $this->template->get();
		$pages 			= $this->page->get();
		$themes_temp 	= scandir(ABSOLUTE_PATH . "/css/themes/colours");
		$themes_temp 	= array_splice($themes_temp, 2);			// Remove . and ..
		$themes 		= array();

		foreach($themes_temp as $theme)
		{
			$themes[] = str_replace(".less", "", $theme);
		}

		$fonts_temp = scandir(ABSOLUTE_PATH . "/css/themes/fonts");
		$fonts_temp = array_splice($fonts_temp, 2);			// Remove . and ..
		$fonts 		= array();

		foreach($fonts_temp as $theme)
		{
			$fonts[] = str_replace(".less", "", $theme);
		}

		$this->data["id"] 			= $id;
		$this->data["site"] 		= $site;
		$this->data["templates"] 	= $templates;
		$this->data["pages"] 		= $pages;
		$this->data["themes"] 		= $themes;
		$this->data["fonts"] 		= $fonts;
		$this->data["domains"] 		= $domains;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/site_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function site_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("site", "template", "redirect"));
		$id = $this->input->post("site_id");

		$site = new Site();
		$site->load($id);
		$site->readPostVars();
		$site->mail_password = ($site->mail_password ? $this->encryption->encrypt($site->mail_password) : null);

		$template = new Template();
		$template->load($site->template_id);

		$config = array();
		$config['upload_path'] 		= ABSOLUTE_PATH . '/temp/';
		$config['allowed_types'] 	= 'gif|jpg|jpeg|png';
		$config['max_size']			= '5242880';	//5MB
		$config['encrypt_name']		= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] 		= true;

		$this->load->library('upload', $config);

		if($this->upload->do_upload("favicon"))
		{
			$upload_data 	= $this->upload->data();

			// Create a multi-size favicon from the logo.
			$this->load->library("Icoconverter");
			$destination 	= ABSOLUTE_PATH . '/templates/' . $template->directory . '/images/favicon.ico';
			$dimensions 	= [[16, 16], [24, 24], [32, 32], [64, 64], [128, 128]];
			$converter 		= new Icoconverter($upload_data["full_path"], $dimensions);
			$converter->saveIco($destination);
			unlink($upload_data["full_path"]);
			// Also copy this favicon to the main images directory for use by the admin area.
			copy($destination, ABSOLUTE_PATH . "/images/favicon.ico");
		}

		if($site->save())
		{
			$this->cache->clean();
			$this->clean_css_cache();
			$this->messages[] = "Site Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Site";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->site($site->site_id);
		}
		else
		{
			$this->sites();
		}
	}

	public function activate($id)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model("site");

		$site = new Site();
		$site->load($id);
		$site->show_landing = 0;

		if($id && $site->save())
		{
			$this->clean_css_cache();
			$this->messages[] = "Your site has been activated! If you did this by mistake, <a href='/admin/deactivate/" . $id . "'>click here</a>.";
		}
		else
		{
			$this->errors[] = "Error Activating Site";
		}

		$this->site($id);
	}

	public function deactivate($id)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->load->model("site");

		$site = new Site();
		$site->load($id);
		$site->show_landing = 1;

		if($id && $site->save())
		{
			$this->clean_css_cache();
			$this->messages[] = "Your site has been hidden behind a landing page. If you did this by mistake, <a href='/admin/activate/" . $id . "'>click here</a>.";
		}
		else
		{
			$this->errors[] = "Error Deactivating Site";
		}

		$this->site($id);
	}

	public function site_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		$this->clean_css_cache();
		$this->generic_delete("site", "sites");
	}
	/* end site functions */

	/* category functions */
	public function categories()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("category", "categories");
	}

	public function category($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("category", "site"));

		$category = new Category();
		$category->load($id);

		$sites = $this->site->get();

		$this->data["id"] = $id;
		$this->data["category"] = $category;
		$this->data["sites"] = $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/category_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function category_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model("category");
		$id = $this->input->post("category_id");

		$category = new Category();
		$category->load($id);
		$category->readPostVars();

		if($category->save())
		{
			$this->cache->clean();
			$this->messages[] = "Category Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Category";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->category($category->category_id);
		}
		else
		{
			$this->categories();
		}
	}

	public function category_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("category", "categories");
	}
	/* end category functions */

	/* page functions */
	public function pages()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_PAGES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("category", "content_type"));

		$content_types 	= $this->content_type->get();
		$categories 	= $this->category->loadBySiteID(Mainframe::active_site_id());

		$this->data["content_types"] 	= $content_types;
		$this->data["categories"] 		= $categories;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/page_list", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function content_types()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_PAGES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("content_type"));

		$content_types = $this->content_type->get();
		$sites = $this->site->get();

		$this->data["content_types"] 	= $content_types;
		$this->data["sites"] 			= $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/content_types", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function page($id=0, $content_type_id=1)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_PAGES) ||
		   !$this->acl->checkPage($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("category", "site", "page", "moduleinstance", "content_type", "page_tag"));

		$page = new Page();
		$page->load($id);

		if(!$id)
		{
			$type = new Content_type();
			$type->load($content_type_id);

			$page->content_type_id 			= $type->content_type_id;
			$page->content_type 			= $type->content_type;
			$page->content_type_filename 	= $type->filename;
			$page->page_info 				= $type->page_info;
		}

		$sites 			= $this->site->get();
		$categories 	= $this->category->get();
		$content_types 	= $this->content_type->get();
		$tags 			= $this->page_tag->get();

		//$modules = new ModuleInstanceList();
		$modules = $this->moduleinstance->LoadInstalledByPageID($page->page_id, ($page->site_id ? $page->site_id : Mainframe::active_site_id()));

		// If this is a new page, we're going to automatically enable modules
		// that are enabled on all published pages already.
		if(!$page->page_id)
		{
			$defaults = $this->moduleinstance->loadDefaultModules(($page->site_id ? $page->site_id : Mainframe::active_site_id()));

			foreach($defaults as $d)
			{
				foreach($modules as $m)
				{
					if($m->module_instance_id == $d->module_instance_id)
					{
						$m->module_installed = 1;
					}
				}
			}
		}

		$acls = $this->acl->loadByPageID($page->page_id);

		$this->data["id"] 			= $id;
		$this->data["page"] 		= $page;
		$this->data["sites"] 		= $sites;
		$this->data["categories"] 	= $categories;
		$this->data["content_types"]= $content_types;
		$this->data["tags"]			= $tags;
		$this->data["modules"] 		= $modules;
		$this->data["acls"]			= $acls;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/page_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function page_save()
	{
		$this->init();
		csrf_verify();
		$id = $this->input->post("page_id");

		if(!$this->current_user->has(ACTION_PAGES) ||
		   !$this->acl->checkPage($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("page", "moduleinstance", "modulepagelink", "content_value", "page_tag"));

		$page = new Page();
		$page->load($id);
		$old_url = $page->url;
		$page->readPostVars();
		$page->start_publishing = (strtotime($this->input->post("start_publishing")) 	!== false ? date("Y-m-d H:i:s", strtotime($this->input->post("start_publishing")))  : null);
		$page->stop_publishing 	= (strtotime($this->input->post("stop_publishing")) 	!== false ? date("Y-m-d H:i:s", strtotime($this->input->post("stop_publishing"))) 	: null);

		$count = 1;
		$new_url = $page->url;

		// Loop until we have a unique URL for this page/site.
		while(!$this->page->checkURL(Mainframe::active_site_id(), $page->page_id, $new_url))
		{
			$new_url = $page->url . $count;
			$count++;
		}

		// If the URL was modified above, set it to the new URL.
		$page->url = $new_url;

		if($page->created == "" || $page->created == "0000-00-00 00:00:00")
		{
			$page->created = date("Y-m-d H:i:s");
		}
		else
		{
			$page->modified = date("Y-m-d H:i:s");
		}
		$saved = $page->save();
		$this->cache->clean();
		$this->page->checkPublishedDates();

		// Save ACL
		$roles = $this->acl_role->get();

		foreach($roles as $role)
		{
			$acl = new ACL();
			$acl->loadByPageRoleID($page->page_id, $role->acl_role_id);

			if(!$acl->acl_id)
			{
				$acl->page_id 		= $page->page_id;
				$acl->acl_role_id 	= $role->acl_role_id;
			}

			if($this->input->post("acl_" . $role->acl_role_id) == "r")
			{
				$acl->read = 1;
				$acl->write = 0;
			}
			else if($this->input->post("acl_" . $role->acl_role_id) == "rw")
			{
				$acl->read = 1;
				$acl->write = 1;
			}
			else
			{
				$acl->read = 0;
				$acl->write = 0;
			}
			$acl->save();
		}

		$modules = $this->moduleinstance->LoadInstalledByPageID($page->page_id, $page->site_id);

		foreach($modules as $module)
		{
			$mp = new ModulePageLink();
			$new = false;

			if(!$mp->LoadByModulePageIDs($module->module_instance_id, $page->page_id))
			{
				/*
				no longer needed
				if($this->input->post("module_$module->module_instance_id") == "1")
				{
					$mp->Add();
				}
				*/

				$new = true;
			}

			if($this->input->post("module_$module->module_instance_id") == "1")
			{
				$mp->module_instance_id = $module->module_instance_id;
				$mp->page_id = $page->page_id;
				$saved = $mp->save() && $saved;
			}
			else if(!$new)
			{
				$saved = $mp->Delete() && $saved;
			}
		}

		// Using the new content types, we just look for any POST vars starting with field_
		// Look for textual fields.
		// Because unchecked checkboxes are NOT sent in POST, we also have to remove any settings that were not sent.
		$content_value_ids = "";
		foreach($this->input->post() as $key => $value)
		{
			if(preg_match('/^field_/', $key))
			{
				$key = str_replace("field_", "", $key);

				// Data URLs will be added to the $_FILES array for processing below.
				if(preg_match('/^data:(.*?);/', $value, $filetype))
				{
					$newfile  = array("simulated" => true);
					$filetype = $filetype[1];

					if($filetype == "image/jpeg" || $filetype == "image/jpg")
					{
						$newfile["file_name"] = $page->page_id . "_" . $key . ".jpg";
						$newfile["file_ext"] = "jpg";
					}
					else if($filetype == "image/png")
					{
						$newfile["file_name"] = $page->page_id . "_" . $key . ".png";
						$newfile["file_ext"] = "png";
					}
					else
					{
						// Only JPG and PNG are supported via data URI.
						continue;
					}

					$newfile["raw_name"] = substr($newfile["file_name"], 0, -4);
					$contents            = file_get_contents($value);
					$tmp_name            = ABSOLUTE_PATH . '/images/content/' . $newfile["file_name"];
					file_put_contents($tmp_name, $contents);
					$_FILES["field_" . $key] = $newfile;

					continue;
				}

				$v = new Content_value();
				$v->LoadByPageKey($page->page_id, $key);
				$v->page_id 	= $page->page_id;
				$v->key 		= $key;
				$v->value 		= cleanAngularGarbage($value);

				if($key == "page_id")
				{
					$v->value 	= $page->page_id;
				}

				$saved = $v->save() && $saved;
				$content_value_ids .= $v->content_value_id . ",";
			}
		}

		if($content_value_ids)
		{
			$content_value_ids = substr($content_value_ids, 0, -1);
			$this->content_value->delete_all_others($page->page_id, $content_value_ids);
		}
		else
		{
			$this->content_value->delete_all($page->page_id);
		}

		// Look for file uploads.
		$config['upload_path'] 		= ABSOLUTE_PATH . '/images/content/';
		$config['allowed_types'] 	= 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx';
		$config['encrypt_name']		= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] 		= true;
		$this->load->library('upload', $config);

		foreach($_FILES as $key => $value)
		{
			if(preg_match('/^field_/', $key))
			{
				if(isset($_FILES[$key]["simulated"]) || $this->upload->do_upload("$key"))
				{
					if(isset($_FILES[$key]["simulated"]))
					{
						$upload_data = $_FILES[$key];
					}
					else
					{
						$upload_data = $this->upload->data();
					}
					$key      = str_replace("field_", "", $key);
					$filename = $upload_data["file_name"];

					Mainframe::deleteThumbnails($upload_data["full_path"]);
					Mainframe::deleteThumbnails(ABSOLUTE_PATH . "/images/content/" . $filename);

					// If there are width and height fields with a matching name, we'll resize the image automatically.
					// ie: if the image field was "myimage" we look for "myimage_width" and "myimage_height"
					if($this->input->post("field_" . $key . "_width") && $this->input->post("field_" . $key . "_height"))
					{
						$filename = $upload_data["raw_name"] . "_" . $this->input->post("field_" . $key . "_width") . "x" . $this->input->post("field_" . $key . "_height") . $upload_data["file_ext"];

						// Delete file first, if it exists.
						if(file_exists(ABSOLUTE_PATH . "/images/content/" . $filename))
						{
							unlink(ABSOLUTE_PATH . "/images/content/" . $filename);
						}

						if($this->input->post("field_" . $key . "_crop") != "")
						{
							$crop = $this->input->post("field_" . $key . "_crop");

							if($crop == "0")
							{
								$crop = false;
							}
						}
						else
						{
							$crop = $this->input->post("field_" . $key . "_width") . ":" . $this->input->post("field_" . $key . "_height");
						}

						createThumbnail($upload_data["full_path"], ABSOLUTE_PATH . "/images/content/" . $filename,
						                $this->input->post("field_" . $key . "_width"),
						                $this->input->post("field_" . $key . "_height"), true,
						                $crop, Mainframe::site()->webp ? IMAGETYPE_WEBP : null);
					}

					$v = new Content_value();
					$v->LoadByPageKey($page->page_id, $key);
					$v->page_id 	= $page->page_id;
					$v->key 		= $key;
					$v->value 		= "/images/content/" . $filename;
					$saved 			= $v->save() && $saved;
				}
			}
		}

		if($saved)
		{
			// Tags
			$ids = "";
			$tags = explode(",", $this->input->post("tag_ids"));

			foreach($tags as $t)
			{
				$tag = new Page_tag();

				if(is_numeric($t))
				{
					$tag->load($t);
				}
				else if(trim($t) != "")
				{
					$t = trim($t);
					$tag->loadByTagName($t);
					$tag->tag = $t;
					$tag->save();
				}

				if($tag->tag_id)
				{
					$this->page_tag->add_link($tag->tag, $page->page_id);
					$ids .= $tag->tag_id . ",";
				}
			}

			if($ids)
			{
				$ids = substr($ids, 0, -1);
				$this->page_tag->deleteAllOtherTags($page->page_id, $ids);
			}
			else
			{
				$this->page_tag->deleteAllTags($page->page_id);
			}

			$this->clean_css_cache();
			$this->cache->clean();
			$this->messages[] = "Page Saved";

			// Reload the page so we get reloaded field values
			$page->load($page->page_id);

			foreach($page->fields as $f)
			{
				$width = null;
				$height = null;

				foreach($page->fields as $f2)
				{
					if($f2->key == $f->key . "_width")
					{
						$width = $f2->value;
					}
					else if($f2->key == $f->key . "_height")
					{
						$height = $f2->value;
					}
				}

				if($width && $height)
				{
					// If this is already a resized image, which it should be, get the original file.
					if(preg_match('/_([0-9]*)x([0-9]*)\.(jpe?g|png|gif|webp)/i', $f->value))
					{
						$src = str_replace(LIVE_SITE, "", preg_replace('/_([0-9]*)x([0-9]*)\.(jpe?g|png|gif|webp)/i', '.$3', $f->value));
					}
					else
					{
						$src = $f->value;
					}

					if($src && file_exists(ABSOLUTE_PATH . $src))
					{
						$ext 		= strrpos($src, ".");
						$filename 	= substr($src, 0, $ext) . "_" . $width . "x" . $height . substr($src, $ext);

						// Delete file first, if it exists.
						if(file_exists(ABSOLUTE_PATH . $filename))
						{
							unlink(ABSOLUTE_PATH . $filename);
						}
						Mainframe::deleteThumbnails(ABSOLUTE_PATH . $src);
						Mainframe::deleteThumbnails(ABSOLUTE_PATH . $filename);

						createThumbnail(ABSOLUTE_PATH . $src,
						                ABSOLUTE_PATH . $filename,
							            $width,
							            $height, true,
							            $width . ":" . $height,
							            Mainframe::site()->webp ? IMAGETYPE_WEBP : null);

						$v = new Content_value();
						$v->Load($f->content_value_id);
						$v->value = $filename;
						$v->save();
					}
				}
			}

			if($page->content_type_id == 1) // Blank page
			{
				// Get all images from the content.
				preg_match_all('/<img (.*?)\/?>/i', $page->field("content1"), $images);

				// Remove any which are module placeholders.
				$final_images = array();
				foreach($images[0] as $i)
				{
					if(stripos($i, "module_placeholder") === false)
					{
						$final_images[] = $i;
					}
				}

				// If only one image is present, the user should consider a "content + image" content type.
				if(count($final_images) == 1)
				{
					$this->messages[] = "Your content only has one image. The content type \"Content with image aligned left or right\" may be better suited. <a href='/admin/content_type_1_to_2/" . $page->page_id . "'>Click here to automatically convert it.</a>";
				}
			}

			// If the URL has been updated, we'll automatically add a redirect for the old URL.
			if($old_url && $old_url != $new_url)
			{
				$redirect = new Redirect();
				$redirect->loadByURL($old_url, $page->site_id);
				$redirect->old_url = $old_url;
				$redirect->new_url = null;
				$redirect->site_id = $page->site_id;
				$redirect->page_id = $page->page_id;
				$redirect->save();

				$this->messages[] = "A redirect was automatically created because you changed the page's URL.";
			}
		}
		else
		{
			$this->errors[] = "Error Saving Page";
		}

		// Regenerate RSS feed.
		$this->generate_rss();

		if($this->input->post("submit_action") == "continue")
		{
			$this->page($page->page_id);
		}
		else
		{
			$this->pages();
		}
	}

	public function page_delete()
	{
		$this->init();
		$id = get_id();

		if(!$this->current_user->has(ACTION_PAGES) ||
		   !$this->acl->checkPage($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->clean_css_cache();
		$this->generic_delete("page", "pages");
	}

	public function content_type_1_to_2($page_id)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_PAGES) ||
		   !$this->acl->checkPage($page_id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$p = new Page();
		$p->load($page_id);

		// Get all images from the content.
		preg_match('/<img (.*?)\/?>/i', $p->field("content1"), $images);

		if(count($images) == 0)
		{
			$this->errors[] = "No images were found in this content item. It was not converted.";
			$this->page($p->page_id);
			return;
		}

		// Get the image src
		preg_match('/<img .*src=[\'"](.*?)[\'"].*>/', $images[0], $srcs);

		$copy_from 	= str_replace("//", "/", ABSOLUTE_PATH . "/" . str_replace(LIVE_SITE, "", $srcs[1]));
		$copy_to 	= ABSOLUTE_PATH . "/images/content/" . basename($srcs[1]);
		$size 		= getimagesize($copy_from);

		if(!copy($copy_from, $copy_to))
		{
			$this->errors[] = "An error occurred converting this page.";
			$this->page($p->page_id);
			return;
		}

		$p->content_type_id = 2;
		$p->save();
		$this->cache->clean();

		$setting = new Content_value();
		$setting->load($p->fields["content1"]->content_value_id);
		$setting->value = preg_replace('/<img (.*?)\/?>/', "", $setting->value);
		$setting->save();

		$setting = new Content_value();
		$setting->page_id = $p->page_id;
		$setting->key = "image";
		$setting->value = "/images/content/" . basename($srcs[1]);
		$setting->save();

		$setting = new Content_value();
		$setting->page_id = $p->page_id;
		$setting->key = "image_width";
		$setting->value = $size[0];
		$setting->save();

		$setting = new Content_value();
		$setting->page_id = $p->page_id;
		$setting->key = "image_height";
		$setting->value = $size[1];
		$setting->save();

		$setting = new Content_value();
		$setting->page_id = $p->page_id;
		$setting->key = "image-alignment";
		$setting->value = "right";
		$setting->save();

		$this->cache->clean();
		$this->messages[] = "Your page was converted to \"Content with image aligned left or right\"";
		$this->page($p->page_id);
	}

	public function generate_rss()
	{
		$CI =& get_instance();
		$CI->load->model(array("site", "page"));

		// Currently RSS is only supported for the first site, which 99.9% of the time is the only site.
		$site = new Site();
		$site->load(1);
		$pages = $CI->page->LoadForRSSFeed(1);

		$rss = '<?xml version="1.0" encoding="UTF-8" ?>' . "\n";
		$rss .= '<rss version="2.0">' . "\n";
		$rss .= "\t" . '<channel>' . "\n";
		$rss .= "\t\t" . '<title>' . htmlspecialchars($site->title) . '</title>' . "\n";
		$rss .= "\t\t" . '<description></description>' . "\n";
 		$rss .= "\t\t" . '<link>' . LIVE_SITE . '</link>' . "\n";
 		$rss .= "\t\t" . '<lastBuildDate>' . date("D, d M Y H:i:s O") . '</lastBuildDate>' . "\n";
 		$rss .= "\t\t" . '<pubDate>' . date("D, d M Y H:i:s O") . '</pubDate>' . "\n";
 		$rss .= "\t\t" . '<ttl>1800</ttl>' . "\n";

 		foreach($pages as $p)
 		{
 			$rss .= "\t\t" . '<item>' . "\n";
			$rss .= "\t\t\t" . '<title>' . htmlspecialchars($p->title) . '</title>' . "\n";
			$rss .= "\t\t\t" . '<description>' . htmlspecialchars($p->description) . '</description>' . "\n";
			$rss .= "\t\t\t" . '<link>' . LIVE_SITE . "/" . $p->url . '</link>' . "\n";
			$rss .= "\t\t\t" . '<guid isPermaLink="false">' . $p->page_id . '</guid>' . "\n";
			$rss .= "\t\t\t" . '<pubDate>' . date("D, d M Y H:i:s O", strtotime($p->created)) . '</pubDate>' . "\n";
			$rss .= "\t\t" . '</item>' . "\n";
 		}

 		$rss .= "\t" . '</channel>' . "\n";
		$rss .= '</rss>';

		@file_put_contents(ABSOLUTE_PATH . "/rss.xml", $rss);
	}
	/* end page functions */

	/* module functions */
	public function modules_available()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES))
		{
			return $this->denied($this->data);
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/module_list_available", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function modules()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES))
		{
			return $this->denied($this->data);
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/module_list", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function module($module_id, $module_instance_id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES) ||
		   !$this->acl->checkModule($module_instance_id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("module", "moduleinstance", "site", "page"));

		$module = new ModuleInstance();
		$module->Load($module_instance_id);

		$sites 		= $this->site->get();
		$module_id 	= (isset($module->module_id) ? $module->module_id : $module_id);
		$pages 		= $this->page->LoadBySiteModuleIDs(($module->site_id ? $module->site_id : Mainframe::active_site_id()), $module_instance_id, $this->current_user);

		$module_type = new Module();
		$module_type->Load($module_id);

		// When creating new modules, the file_name is not set.  We need it.
		$module->module_id = $module_id;
		$module->file_name = $module_type->file_name;

		$site = new Site();
		$site->Load(($module->site_id ? $module->site_id : Mainframe::active_site_id()));

		require_once(MODULE_PATH . "/$module_type->file_name/$module_type->file_name" . ".php");
		$m = new $module_type->class_name;
		$m->Load($module_instance_id);

		$modules 	= Mainframe::LoadModulesBySiteID(Mainframe::active_site_id());
		$positions 	= Mainframe::GetModulePositions($site->template_id);
		$acls 		= $this->acl->loadByModuleInstanceID($module_instance_id);

		$this->data["id"] 			= $module_instance_id;
		$this->data["module"] 		= $module;
		$this->data["positions"] 	= $positions;
		$this->data["pages"] 		= $pages;
		$this->data["site"] 		= $site;
		$this->data["sites"] 		= $sites;
		$this->data["m"] 			= $m;
		$this->data["modules"] 		= $modules;
		$this->data["module_id"] 	= $module_id;
		$this->data["acls"]			= $acls;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/module_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function module_save()
	{
		$this->init();
		$id = get_id();
		csrf_verify();

		if(!$this->current_user->has(ACTION_MODULES) ||
		   !$this->acl->checkModule($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("module", "modulesetting", "moduleinstance", "page", "modulepagelink"));

		$module = new ModuleInstance();
		$module->load($id);
		$module->readPostVars();

		if($module->position == "")
		{
			$module->always_on = 0;
		}

		$saved = $module->save();
		$this->cache->clean();

		// Save ACL
		$roles = $this->acl_role->get();

		foreach($roles as $role)
		{
			$acl = new ACL();
			$acl->loadByModuleRoleID($module->module_instance_id, $role->acl_role_id);

			if(!$acl->acl_id)
			{
				$acl->module_instance_id 	= $module->module_instance_id;
				$acl->acl_role_id 			= $role->acl_role_id;
			}

			if($this->input->post("acl_" . $role->acl_role_id) == "r")
			{
				$acl->read = 1;
				$acl->write = 0;
			}
			else if($this->input->post("acl_" . $role->acl_role_id) == "rw")
			{
				$acl->read = 1;
				$acl->write = 1;
			}
			else
			{
				$acl->read = 0;
				$acl->write = 0;
			}
			$acl->save();
		}

		$m1 = new Module();
		$m1->load($module->module_id);

		// Run the module's installer, which will be configured to handle module dependencies if required by this module.
		require_once(MODULE_PATH . "/" . $m1->file_name . "/" . $m1->file_name . ".php");

		$m2 = new $m1->class_name();
		$m2->Load($m1->module_id);
		try
		{
			$m2->install();
		}
		catch(Exception $e){}

		$pages = $this->page->LoadBySiteModuleIDs($this->input->post("site_id"), $module->module_instance_id, $this->current_user);

		foreach($pages as $page)
		{
			$mp = new ModulePageLink();
			$new = false;

			if(!$mp->LoadByModulePageIDs($module->module_instance_id, $page->page_id))
			{
				/*
				no longer needed
				if($this->input->post("page_$page->page_id") == "1")
				{
					$mp->Add();
				}
				*/

				$new = true;
			}

			if($module->always_on > 0 || $this->input->post("page_$page->page_id") == "1")
			{
				// Skip the default page if always_on == "all pages except home"
				if($module->always_on == 2 && Mainframe::site()->default_page_id == $page->page_id)
				{
					// If the link already existed, delete it.
					if(!$new)
					{
						$mp->delete();
					}

					continue;
				}
				// Skip all inside pages if always_on == "only main page"
				else if($module->always_on == 3 && Mainframe::site()->default_page_id != $page->page_id)
				{
					// If the link already existed, delete it.
					if(!$new)
					{
						$mp->delete();
					}

					continue;
				}

				$mp->module_instance_id = $module->module_instance_id;
				$mp->page_id = $page->page_id;
				$saved = $mp->save() && $saved;
			}
			else if(!$new)
			{
				$saved = $mp->Delete() && $saved;
			}
		}

		// Using the new module_settings, we just look for any POST vars starting with param_
		// Because unchecked checkboxes are NOT sent in POST, we also have to remove any settings that were not sent.
		// First we are going to check for param_ which will always be the current module.
		$ids = "";

		foreach($this->input->post() as $key => $value)
		{
			if(preg_match('/^param_/', $key))
			{
				$key = str_replace("param_", "", $key);

				$s = new ModuleSetting();
				$s->LoadByModuleKey($module->module_instance_id, $key);
				$s->module_instance_id 	= $module->module_instance_id;
				$s->key 				= $key;
				$s->value 				= cleanAngularGarbage($value);
				$saved 					= $s->save() && $saved;

				$ids .= $s->module_setting_id . ",";
			}
		}

		// Look for file uploads.
		$config['upload_path'] 		= ABSOLUTE_PATH . '/images/content/';
		$config['allowed_types'] 	= 'gif|jpg|jpeg|png|pdf|doc|docx|xls|xlsx';
		$config['encrypt_name']		= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] 		= true;
		$this->load->library('upload', $config);

		foreach($_FILES as $key => $value)
		{
			if(preg_match('/^param_/', $key))
			{
				if($this->upload->do_upload("$key"))
				{
					$key 			= str_replace("param_", "", $key);
					$upload_data 	= $this->upload->data();
					$filename 		= $upload_data["file_name"];

					Mainframe::deleteThumbnails($upload_data["full_path"]);
					Mainframe::deleteThumbnails(ABSOLUTE_PATH . "/images/content/" . $filename);

					// If there are width and height fields with a matching name, we'll resize the image automatically.
					// ie: if the image field was "myimage" we look for "myimage_width" and "myimage_height"
					if($this->input->post("param_" . $key . "_width") && $this->input->post("param_" . $key . "_height"))
					{
						$filename = $upload_data["raw_name"] . "_" . $this->input->post("param_" . $key . "_width") . "x" . $this->input->post("param_" . $key . "_height") . $upload_data["file_ext"];

						// Delete file first, if it exists.
						if(file_exists(ABSOLUTE_PATH . "/images/content/" . $filename))
						{
							unlink(ABSOLUTE_PATH . "/images/content/" . $filename);
						}

						if($this->input->post("param_" . $key . "_crop") != "")
						{
							$crop = $this->input->post("param_" . $key . "_crop");

							if($crop == "0")
							{
								$crop = false;
							}
						}
						else
						{
							$crop = $this->input->post("param_" . $key . "_width") . ":" . $this->input->post("param_" . $key . "_height");
						}

						createThumbnail($upload_data["full_path"], ABSOLUTE_PATH . "/images/content/" . $filename,
						                $this->input->post("param_" . $key . "_width"),
						                $this->input->post("param_" . $key . "_height"), true,
						                $crop, Mainframe::site()->webp ? IMAGETYPE_WEBP : null);
					}

					$s = new ModuleSetting();
					$s->LoadByModuleKey($module->module_instance_id, $key);
					$s->module_instance_id 	= $module->module_instance_id;
					$s->key 				= $key;
					$s->value 				= "/images/content/" . $filename;
					$saved 					= $s->save() && $saved;
					$ids .= $s->module_setting_id . ",";
				}
			}
		}

		if($ids)
		{
			$ids = substr($ids, 0, -1);
			$this->modulesetting->delete_all_others($module->module_instance_id, $ids);
		}
		else
		{
			$this->modulesetting->delete_all($module->module_instance_id);
		}

		// If there are any children of this module, their fields will be indexed starting from 0, in the same order as they
		// were defined in the module. For example: param_1_width refers to "width" for "child 1" of this module (index 0).
		// The form indexes are 1 based while the children array is zero based. This is for clarity on the settings form.
		if(count($m2->children()))
		{
			for($i=0; $i<count($m2->children()); $i++)
			{
				$module_instance_id 	= $m2->children()[$i];
				$ids 			= "";

				foreach($this->input->post() as $key => $value)
				{
					if(preg_match('/^param_' . ($i+1) . '_/', $key))
					{
						$key = str_replace("param_" . ($i+1) . "_", "", $key);

						$s = new ModuleSetting();
						$s->LoadByModuleKey($module_instance_id, $key);
						$s->module_instance_id 	= $module_instance_id;
						$s->key 				= $key;
						$s->value 				= cleanAngularGarbage($value);
						$saved 					= $s->save() && $saved;

						$ids .= $s->module_setting_id . ",";
					}
				}

				if($ids)
				{
					$ids = substr($ids, 0, -1);
					$this->modulesetting->delete_all_others($module_instance_id, $ids);
				}
				else
				{
					$this->modulesetting->delete_all($module_instance_id);
				}
			}
		}

		if($saved)
		{
			$this->clean_css_cache();
			$this->cache->clean();
			$this->messages[] = "Module Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Module";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->module($module->module_id, $module->module_instance_id);
		}
		else
		{
			$this->modules();
		}
	}

	public function module_delete()
	{
		$this->init();
		$id = get_id();

		if(!$this->current_user->has(ACTION_MODULES) ||
		   !$this->acl->checkModule($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model("moduleinstance");

		if($id)
		{
			$moduleinstance = new ModuleInstance();
			$moduleinstance->load($id);

			if($moduleinstance->delete())
			{
				$this->clean_css_cache();
				$this->messages[] = "Module Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Module.";
			}
		}

		$this->modules();
	}

	public function module_duplicate($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES))
		{
			return $this->denied($this->data);
		}

		if($id <= 0)
		{
			$this->errors[] = "Invalid Module";
			$this->modules_available();
			return;
		}

		$this->load->model("module");

		$module = new Module();
		$module->load($id);

		$new_module = new Module();
		$new_module->module_id 	= "";
		$new_module->name 		= $module->name . " (Custom)";
		$new_module->file_name 	= $module->file_name . "_custom";
		$new_module->class_name = "Custom" . $module->class_name;

		// Keep going until we find a directory name that doesn't exist yet.
		// In most cases, this loop will never run.
		$count = 0;
		while(file_exists(ABSOLUTE_PATH . "/modules/" . $new_module->file_name))
		{
			$count++;
			$new_module->file_name = $module->file_name . "_custom" . $count;
			$new_module->class_name = "Custom" . $module->class_name . $count;
		}
		$new_module->save();

		// Copy main module directory.
		if(file_exists(ABSOLUTE_PATH . "/modules/" . $module->file_name))
		{
			copy_directory(ABSOLUTE_PATH . "/modules/" . $module->file_name, ABSOLUTE_PATH . "/modules/" . $new_module->file_name);
		}

		// Copy module views directory.
		if(file_exists(ABSOLUTE_PATH . "/application/views/modules/" . $module->file_name))
		{
			copy_directory(ABSOLUTE_PATH . "/application/views/modules/" . $module->file_name, ABSOLUTE_PATH . "/application/views/modules/" . $new_module->file_name);
		}

		// Rename the main module PHP file.
		if(file_exists(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $module->file_name . ".php"))
		{
			rename(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $module->file_name . ".php",
			       ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $new_module->file_name . ".php");
		}

		// Rename the main module LESS file.
		if(file_exists(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $module->file_name . ".less"))
		{
			rename(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $module->file_name . ".less",
			       ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $new_module->file_name . ".less");
		}

		// Automatically replace some parts of the new module.
		$module_content = file_get_contents(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $new_module->file_name . ".php");
		$module_content = str_ireplace("class " . $module->class_name, "class " . $new_module->class_name, $module_content);
		$module_content = str_ireplace("function " . $module->class_name, "function " . $new_module->class_name, $module_content);
		$module_content = str_ireplace("modules/" . $module->file_name, "modules/" . $new_module->file_name, $module_content);
		file_put_contents(ABSOLUTE_PATH . "/modules/" . $new_module->file_name . "/" . $new_module->file_name . ".php", $module_content);

		$this->messages[] = "\"" . $module->name . "\" Duplicated As \"" . $new_module->name . "\"";
		$this->messages[] = "Customize the module at " . ABSOLUTE_PATH . "/modules/" . $new_module->file_name;

		$this->modules_available();
	}
	/* end module functions */

	/* menu functions */
	public function menus()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MENUS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("menu", "menus");
	}

	private function buildMenu($menu_id, $parent_id=null)
	{
		$items = $this->menuitem->LoadByMenuID($menu_id, $parent_id, null);

		foreach($items as $item)
		{
			$item->children = $this->buildMenu($menu_id, $item->menu_item_id);
		}

		return $items;
	}

	public function menu($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MENUS))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("menu", "menuitem", "site", "page"));

		$menu = new Menu();
		$menu->load($id);

		$menuitems 	= $this->buildMenu($id, null);
		$orphans 	= $this->page->getUnlinkedByMenuID(Mainframe::active_site_id(), $menu->menu_id);
		$sites 		= $this->site->get();

		$this->data["id"] 		= $id;
		$this->data["menu"] 	= $menu;
		$this->data["items"] 	= $menuitems;
		$this->data["orphans"] 	= $orphans;
		$this->data["sites"] 	= $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/menu_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	private function _sort_menu_items($sorting, $parent_id)
	{
		$count = 1;

		foreach($sorting as $sort)
		{
			$item = new MenuItem();
			$item->Load($sort->id);
			$item->parent_id = $parent_id;
			$item->sort = $count;
			$item->save();
			$count++;

			if(count($sort->children[0]))
			{
				$this->_sort_menu_items($sort->children[0], $sort->id);
			}
		}
		$this->cache->clean();
	}

	public function menu_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_MENUS))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("menu", "menuitem"));
		$id = $this->input->post("menu_id");

		$menu = new Menu();
		$menu->load($id);
		$menu->readPostVars();

		if($menu->save())
		{
			$this->cache->clean();

			if($this->input->post("sorting"))
			{
				$sorting = json_decode($this->input->post("sorting"))[0];
				$this->_sort_menu_items($sorting, null);
			}

			$this->cache->clean();
			$this->messages[] = "Menu Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Menu";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->menu($menu->menu_id);
		}
		else
		{
			$this->menus();
		}
	}

	public function menu_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MENUS))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("menu", "menus");
	}
	/* end menu functions */

	/* menu item functions */
	public function menuitem($menu_id, $id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MENUS) ||
		   !$this->acl->checkMenuItem($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("menu", "menuitem", "site", "page"));

		$item = new MenuItem();
		$item->load($id);

		$menus = $this->menu->LoadBySiteID(Mainframe::active_site_id());
		$sites = $this->site->get();

		$menu = new Menu();
		$menu->Load($item->menu_id);

		$items = $this->menuitem->LoadByMenuID(($id ? $item->menu_id : $menu_id), -1, null);
		$pages = $this->page->LoadBySiteID(($menu->site_id ? $menu->site_id : Mainframe::active_site_id()), $this->current_user, "category_name, p.`title`");
		$acls  = $this->acl->loadByMenuItemID($item->menu_item_id);

		$this->data["id"] 		= $id;
		$this->data["menu_id"] 	= $menu_id;
		$this->data["item"] 	= $item;
		$this->data["items"] 	= $items;
		$this->data["menu"] 	= $menu;
		$this->data["menus"] 	= $menus;
		$this->data["sites"] 	= $sites;
		$this->data["pages"] 	= $pages;
		$this->data["acls"] 	= $acls;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/menuitem_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function menuitem_save()
	{
		$this->init();
		csrf_verify();

		$id = $this->input->post("menu_item_id");

		if(!$this->current_user->has(ACTION_MENUS) ||
		   !$this->acl->checkMenuItem($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model("menuitem");

		$item = new MenuItem();
		$item->load($id);
		$item->readPostVars();

		// When no page or URL is provided, link to # automatically.
		// If a page is provided, remove the URL if it was set to #.
		if(!$item->page_id && !$item->url)
		{
			$item->url = "#";
		}
		else if($item->page_id && $item->url == "#")
		{
			$item->url = null;
		}

		if($item->save())
		{
			$this->cache->clean();

			// Save ACL
			$roles = $this->acl_role->get();

			foreach($roles as $role)
			{
				$acl = new ACL();
				$acl->loadByMenuItemRoleID($item->menu_item_id, $role->acl_role_id);

				if(!$acl->acl_id)
				{
					$acl->menu_item_id 	= $item->menu_item_id;
					$acl->acl_role_id 	= $role->acl_role_id;
				}

				if($this->input->post("acl_" . $role->acl_role_id) == "r")
				{
					$acl->read = 1;
					$acl->write = 0;
				}
				else if($this->input->post("acl_" . $role->acl_role_id) == "rw")
				{
					$acl->read = 1;
					$acl->write = 1;
				}
				else
				{
					$acl->read = 0;
					$acl->write = 0;
				}
				$acl->save();
			}

			$this->cache->clean();
			$this->messages[] = "Menu Item Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Menu Item";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->menuitem($item->menu_id, $item->menu_item_id);
		}
		else
		{
			$this->menu($item->menu_id);
		}
	}

	public function menuitem_bulk_save()
	{
		$this->init();
		csrf_verify();

		$ids 		= explode(",", $this->input->post("page_ids"));
		$menu_id 	= $this->input->post("menu_id");
		$sort 		= 1000;

		$this->load->model(array("menuitem", "page", "acl"));

		foreach($ids as $id)
		{
			$p = new Page();
			$p->load($id);

			$item 				= new MenuItem();
			$item->menu_id 		= $menu_id;
			$item->title 		= $p->title;
			$item->page_id 		= $p->page_id;
			$item->sort 		= $sort;
			$item->published 	= 1;
			$sort++;

			if($item->save())
			{
				// Save ACL
				$roles = $this->acl_role->get();

				foreach($roles as $role)
				{
					$acl 				= new Acl();
					$acl->acl_role_id 	= $role->acl_role_id;
					$acl->menu_item_id 	= $item->menu_item_id;

					if($role->acl_role_id == 3)
					{
						$acl->read = 1;
						$acl->write = 1;
					}
					else
					{
						$acl->read = 1;
						$acl->write = 0;
					}
					$acl->save();
				}

				$this->messages[] = "Menu Items Saved";
			}
			else
			{
				$this->errors[] = "Error saving one or more menu items";
			}
		}

		$this->cache->clean();
		$this->messages = array_unique($this->messages);
		$this->errors 	= array_unique($this->errors);

		$this->menu($menu_id);
	}

	public function menuitem_delete()
	{
		$this->init();

		$id = get_id();

		if(!$this->current_user->has(ACTION_MENUS) ||
		   !$this->acl->checkMenuItem($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model("menuitem");

		if($id)
		{
			$item = new MenuItem();
			$item->load($id);
			$menu_id = $item->menu_id;

			if($item->delete())
			{
				$this->messages[] = "Menu Item Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Menu Item.";
			}
		}

		$this->menu($menu_id);
	}
	/* end menu item functions */

	/* booking functions */
	public function appointments()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_APPOINTMENTS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("appointment", "appointments");
	}

	public function appointment_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_APPOINTMENTS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("appointment");

		$id = get_id();

		if($id)
		{
			$appt = new Appointment();
			$appt->load($id);

			if($appt->delete())
			{
				$this->messages[] = "Appointment Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Appointment.";
			}
		}

		$this->appointments();
	}
	/* end booking functions */

	/* event functions */
	public function events()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENTS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("event", "events");
	}

	public function event($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENTS))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("event"));

		$event = new Event();
		$event->load($id);

		$events = $this->event->LoadParents(Mainframe::active_site_id());

		$this->data["id"] = $id;
		$this->data["event"] = $event;
		$this->data["events"] = $events;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/event_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function event_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_EVENTS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("event");
		$id = $this->input->post("event_id");

		$event = new Event();
		$event->load($id);
		$event->readPostVars();

		// URLs on child events are problematic.
		if($event->parent_id > 0)
		{
			$event->url = null;
		}

		$config['upload_path'] = ABSOLUTE_PATH . '/images/events/';
		$config['allowed_types'] = 'gif|jpg|jpeg|png';
		$config['max_size']	= '5242880';	//5MB
		$config['encrypt_name']	= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] = true;
		//$config['max_width']  = '1024';
		//$config['max_height']  = '768';

		$this->load->library('upload', $config);

		if($this->upload->do_upload("banner_upload"))
		{
			$upload_data = array('upload_data' => $this->upload->data());
			$event->banner_image = $upload_data["upload_data"]["file_name"];
		}

		if($event->save())
		{
			$this->cache->clean();
			$this->messages[] = "Event Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Event";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->event($event->event_id);
		}
		else
		{
			$this->events();
		}
	}

	public function event_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENTS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("event");

		$id = get_id();

		if($id)
		{
			$event = new Event();
			$event->load($id);

			if($event->delete())
			{
				$this->messages[] = "Event Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Event.";
			}
		}

		$this->events();
	}
	/* end event functions */

	/* event form functions */
	public function event_forms()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENT_FORMS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("event_form", "event_forms");
	}

	public function event_form($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENT_FORMS))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("event", "event_form"));

		$form = new Event_Form();
		$form->load($id);

		$events = $this->event->LoadParents(Mainframe::active_site_id());

		$this->data["id"] = $id;
		$this->data["form"] = $form;
		$this->data["events"] = $events;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/event_form_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function event_form_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_EVENT_FORMS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("event_form");
		$id = $this->input->post("form_id");

		$form = new Event_Form();
		$form->load($id);
		$form->readPostVars();

		if($form->save())
		{
			$this->cache->clean();
			$this->messages[] = "Event Form Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Event Form";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->event_form($form->form_id);
		}
		else
		{
			$this->event_forms();
		}
	}

	public function event_form_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_EVENT_FORMS))
		{
			return $this->denied($this->data);
		}

		$this->load->model("event_form");

		$id = get_id();

		if($id)
		{
			$form = new Event_Form();
			$form->load($id);

			if($form->delete())
			{
				$this->messages[] = "Event Form Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Event Form";
			}
		}

		$this->event_forms();
	}
	/* end event form functions */

	/* metadata functions */
	public function metadata_pages()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("metadata_page", "page"));

		$id = get_id();

		if($this->input->get_post("add") == "1")
		{
			$metadata = new Metadata_Page();
			$metadata->site_id = Mainframe::active_site_id();
			$metadata->page_id = 0;
			$metadata->url = time();
			$metadata->title = "";
			$metadata->description = "";
			$metadata->save();

			//redirect to avoid refreshes adding more and more new entries
			redirect("/admin/metadata_pages");
		}

		$metadata = $this->metadata_page->LoadAllPagesBySiteID(Mainframe::active_site_id());
		$pages = $this->page->get();

		$this->data["metadata"] = $metadata;
		$this->data["pages"] = $pages;
		$this->data["sites"] = $this->sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/metadata_pages", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function metadata_pages_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("metadata_page", "page"));

		$id = $this->input->post("metadata_id");

		$metadata = new Metadata_Page();
		$metadata->load($id);
		$metadata->readPostVars();

		//if we selected a page, use the URL field to update the page's URL
		if($metadata->page_id)
		{
			$page = new Page();
			$page->load($metadata->page_id);
			$page->url = $metadata->url;
			$page->title = $metadata->title;
			$page->description = $metadata->description;
			$page->save();
		}

		//AJAX request
		if($metadata->save())
		{
			$this->cache->clean();
			die((string)$metadata->metadata_id);
		}
		else
		{
			die("FAILED");
		}
	}

	public function get_page_data()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model("page");

		$id = $this->input->post("page_id");

		$page = new Page();
		$page->load($id);

		//AJAX request
		if(isset($page->page_id))
		{
			header("Content-type: application/json");
			echo(json_encode($page));
		}
		else
		{
			die("FAILED");
		}
	}

	public function metadata_pages_delete($id=null)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model("metadata_page");

		if($id)
		{
			$page = new Metadata_Page();
			$page->load($id);

			if($page->delete())
			{
				$this->messages[] = "Meta Data Deleted";
			}
			else
			{
				$this->errors[] = "Error Deleting Meta Data.";
			}

			$this->metadata_pages();
		}
	}

	private function find_dirs(&$dirs, $directory)
	{
		$dirs[] = $directory;

		$new_files = scandir($directory);

		foreach($new_files as $file)
		{
			if($file != "." && $file != "..")
			{
				if(is_dir("$directory/$file"))
				{
					$this->find_dirs($dirs, "$directory/$file");
				}
			}
		}
	}

	private function images_to_array(&$files, $directory)
	{
		$new_files = scandir($directory);

		foreach($new_files as $file)
		{
			if($file != "." && $file != "..")
			{
				if(is_file("$directory/$file") &&
				   preg_match('/(ui-(bg_|icons_)|@2x\.|_thumb_)/', $file) === 0)
				{
					if(preg_match('/\.(jpg|jpeg|gif|png|bmp|tiff)$/i', trim($file)) > 0)
					{
						$files[] = "$directory/$file";
					}
				}
				// else if(is_dir("$directory/$file"))
				// {
				// 	$this->images_to_array($files, "$directory/$file");
				// }
			}
		}
	}
	/* end metadata functions */

	/* role functions */
	public function roles()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_ROLES))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("role", "roles");
	}

	public function role($acl_role_id=null)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_ROLES))
		{
			return $this->denied($this->data);
		}

		$role = new Acl_Role();
		$role->load($acl_role_id);

		$actions = $this->acl_action->loadByRoleID($role->acl_role_id);

		$this->data["actions"] 		= $actions;
		$this->data["acl_role_id"] 	= $acl_role_id;
		$this->data["role"] 		= $role;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/role_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function role_save()
	{
		$this->init();
		csrf_verify();

		$acl_role_id = $this->input->post("acl_role_id");

		if(!$this->current_user->has(ACTION_ROLES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("acl", "acl_role_action", "page", "moduleinstance", "document", "menuitem"));

		$role = new Acl_Role();
		$role->load($acl_role_id);
		$role->readPostVars();

		$saved = $role->save();
		$this->cache->clean();

		$actions = $this->acl_action->get();

		foreach($actions as $action)
		{
			$checked = $this->input->post("action_" . $action->acl_action_id);

			if($checked)
			{
				$this->acl_role_action->add_link($role->acl_role_id, $action->acl_action_id);
			}
			else
			{
				$this->acl_role_action->delete_link($role->acl_role_id, $action->acl_action_id);
			}
		}

		if($this->input->post("bulk_page_acl") == "n")
		{
			$pages = $this->page->get();

			foreach($pages as $p)
			{
				$acl = new ACL();
				$acl->loadByPageRoleID($p->page_id, $role->acl_role_id);
				$acl->delete();
			}
		}
		else if($this->input->post("bulk_page_acl") == "r")
		{
			$pages = $this->page->get();

			foreach($pages as $p)
			{
				$acl = new ACL();
				$acl->loadByPageRoleID($p->page_id, $role->acl_role_id);
				$acl->page_id = $p->page_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 0;
				$acl->save();
			}
		}
		else if($this->input->post("bulk_page_acl") == "rw")
		{
			$pages = $this->page->get();

			foreach($pages as $p)
			{
				$acl = new ACL();
				$acl->loadByPageRoleID($p->page_id, $role->acl_role_id);
				$acl->page_id = $p->page_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 1;
				$acl->save();
			}
		}

		if($this->input->post("bulk_document_acl") == "n")
		{
			$documents = $this->document->get();

			foreach($documents as $d)
			{
				$acl = new ACL();
				$acl->loadBydocumentRoleID($d->document_id, $role->acl_role_id);
				$acl->delete();
			}
		}
		else if($this->input->post("bulk_document_acl") == "r")
		{
			$documents = $this->document->get();

			foreach($documents as $d)
			{
				$acl = new ACL();
				$acl->loadBydocumentRoleID($d->document_id, $role->acl_role_id);
				$acl->document_id = $d->document_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 0;
				$acl->save();
			}
		}
		else if($this->input->post("bulk_document_acl") == "rw")
		{
			$documents = $this->document->get();

			foreach($documents as $d)
			{
				$acl = new ACL();
				$acl->loadByDocumentRoleID($d->document_id, $role->acl_role_id);
				$acl->document_id = $d->document_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 1;
				$acl->save();
			}
		}

		if($this->input->post("bulk_module_acl") == "n")
		{
			$modules = $this->moduleinstance->get();

			foreach($modules as $m)
			{
				$acl = new ACL();
				$acl->loadByModuleRoleID($m->module_instance_id, $role->acl_role_id);
				$acl->delete();
			}
		}
		else if($this->input->post("bulk_module_acl") == "r")
		{
			$modules = $this->moduleinstance->get();

			foreach($modules as $m)
			{
				$acl = new ACL();
				$acl->loadByModuleRoleID($m->module_instance_id, $role->acl_role_id);
				$acl->module_instance_id = $m->module_instance_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 0;
				$acl->save();
			}
		}
		else if($this->input->post("bulk_module_acl") == "rw")
		{
			$modules = $this->moduleinstance->get();

			foreach($modules as $m)
			{
				$acl = new ACL();
				$acl->loadByModuleRoleID($m->module_instance_id, $role->acl_role_id);
				$acl->module_instance_id = $m->module_instance_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 1;
				$acl->save();
			}
		}

		if($this->input->post("bulk_menu_acl") == "n")
		{
			$menu_items = $this->menuitem->get();

			foreach($menu_items as $m)
			{
				$acl = new ACL();
				$acl->loadByMenuItemRoleID($m->menu_item_id, $role->acl_role_id);
				$acl->delete();
			}
		}
		else if($this->input->post("bulk_menu_acl") == "r")
		{
			$menu_items = $this->menuitem->get();

			foreach($menu_items as $m)
			{
				$acl = new ACL();
				$acl->loadByMenuItemRoleID($m->menu_item_id, $role->acl_role_id);
				$acl->menu_item_id = $m->menu_item_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 0;
				$acl->save();
			}
		}
		else if($this->input->post("bulk_menu_acl") == "rw")
		{
			$menu_items = $this->menuitem->get();

			foreach($menu_items as $m)
			{
				$acl = new ACL();
				$acl->loadByMenuItemRoleID($m->menu_item_id, $role->acl_role_id);
				$acl->menu_item_id = $m->menu_item_id;
				$acl->acl_role_id = $role->acl_role_id;
				$acl->read = 1;
				$acl->write = 1;
				$acl->save();
			}
		}

		$this->cache->clean();

		if(!$acl_role_id)
		{
			//new record, continue editing so we can add actions
			$this->messages[] = "Role Created";
			$this->role($role->acl_role_id);
		}
		else
		{
			if($saved)
			{
				$this->messages[] = "Role Saved";
			}
			else
			{
				$this->errors[] = "Error Saving Role";
			}

			if($this->input->post("submit_action") == "continue")
			{
				$this->role($role->acl_role_id);
			}
			else
			{
				$this->roles();
			}
		}
	}

	public function role_delete($acl_role_id)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_ROLES))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("acl_role", "roles", $acl_role_id);
	}
	/* end role functions */

	/**
	 * Generate a random password and its related hash.
	 * The only purpose of this function is to generate a new password and hash for manually updating the database when a password is lost.
	 */
	public function generate_password()
	{
		$this->load->helper("password");

		$alphabet = "abcdefghijklmnopqrstuwxyzABCDEFGHIJKLMNOPQRSTUWXYZ0123456789";
		$password = "";
		$msg = "";

		for ($i = 0; $i < 8; $i++)
		{
        	mt_srand();
			$n = mt_rand(0, strlen($alphabet)-1);
			$password .= $alphabet[$n];
		}

		echo("<pre>Password: $password</pre>");
		echo("<pre>Hash: " . password_hash($password, PASSWORD_BCRYPT) . "</pre>");
	}

	/* directory functions */
	public function update_directory_latlng()
	{
		$this->init();
		$this->load->model("directory_company");
		$company_id = $this->input->get_post("company_id");
		$latitude = $this->input->get_post("latitude");
		$longitude = $this->input->get_post("longitude");

		if($company_id <= 0 || $latitude == "" || $longitude == "")
		{
			die();
		}

		$c = new Directory_Company();
		$c->load($company_id);
		$c->latitude = $latitude;
		$c->longitude = $longitude;
		$c->save();
	}
	/* end directory functions */

	/* directory category functions */
	public function directory_categories()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("directory_category", "directory_categories");
	}

	public function directory_category($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("directory_category", "site"));

		$category = new Directory_Category();
		$category->load($id);

		$sites = $this->site->get();

		$this->data["id"] = $id;
		$this->data["category"] = $category;
		$this->data["sites"] = $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/directory_category_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function directory_category_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_DIRECTORY_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model("directory_category");

		$id = $this->input->post("category_id");

		$category = new Directory_Category();
		$category->load($id);
		$category->readPostVars();

		$config['upload_path'] = ABSOLUTE_PATH . '/images/directory/categories/';
		$config['allowed_types'] = 'gif|jpg|jpeg|png';
		$config['max_size']	= '5242880';	//5MB
		$config['encrypt_name']	= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] = true;
		//$config['max_width']  = '1024';
		//$config['max_height']  = '768';

		$this->load->library('upload', $config);

		if($this->upload->do_upload("photo_upload"))
		{
			$upload_data = array('upload_data' => $this->upload->data());
			$category->photo = $upload_data["upload_data"]["file_name"];
		}

		if($category->save())
		{
			$this->cache->clean();
			$this->messages[] = "Category Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Category";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->directory_category($category->category_id);
		}
		else
		{
			$this->directory_categories();
		}
	}

	public function directory_category_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_CATEGORIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("directory_category", "directory_categories");
	}
	/* end directory category functions */

	/* directory company functions */
	public function directory_companies()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_COMPANIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("directory_company", "directory_companies");
	}

	public function directory_company($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_COMPANIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("directory_company", "directory_category", "site"));

		$company = new Directory_Company();
		$company->load($id);

		$categories = $this->directory_category->LoadByCompanyID($company->company_id);
		$sites = $this->site->get();

		$this->data["id"] = $id;
		$this->data["company"] = $company;
		$this->data["categories"] = $categories;
		$this->data["sites"] = $sites;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/directory_company_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function directory_company_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_DIRECTORY_COMPANIES))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("directory_company", "directory_category", "directory_category_link"));

		$id = $this->input->post("company_id");

		$company = new Directory_Company();
		$company->load($id);
		$company->readPostVars();

		if($company->added == null || $company->added == "" || $company->added == "0000-00-00 00:00:00")
		{
			$company->added = date("Y-m-d H:i:s");
		}

		//upload logo
		$config['upload_path'] = ABSOLUTE_PATH . '/images/directory/logos/';
		$config['allowed_types'] = 'gif|jpg|jpeg|png';
		$config['max_size']	= '5242880';	//5MB
		$config['encrypt_name']	= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] = true;
		//$config['max_width']  = '1024';
		//$config['max_height']  = '768';

		$this->load->library('upload', $config);

		if($this->upload->do_upload("logo_upload"))
		{
			$upload_data = array('upload_data' => $this->upload->data());
			$company->logo = $upload_data["upload_data"]["file_name"];
		}

		//upload photo
		$config['upload_path'] = ABSOLUTE_PATH . '/images/directory/companies/';

		$this->upload->initialize($config);

		if($this->upload->do_upload("photo_upload"))
		{
			$upload_data = array('upload_data' => $this->upload->data());
			$company->photo = $upload_data["upload_data"]["file_name"];
		}

		$saved = $company->save();
		$this->cache->clean();

		$categories = $this->directory_category->LoadBySiteID(Mainframe::active_site_id());

		foreach($categories as $category)
		{
			$link = new Directory_Category_Link();
			$link->LoadByCompanyCategoryIDs($company->company_id, $category->category_id);

			if($this->input->post("category_" . $category->category_id) == "1" && $this->input->post("category_" . $category->category_id . "_current") != "1")
			{
				if(isset($link->directory_company_category_id) && $link->directory_company_category_id > 0)
				{
					//this link already exists
				}
				else
				{
					//create a link
					$link = new Directory_Category_Link();
					$link->company_id = $company->company_id;
					$link->category_id = $category->category_id;
					$saved = $link->save() && $saved;
				}
			}
			else if($this->input->post("category_" . $category->category_id) != "1" && $this->input->post("category_" . $category->category_id . "_current") == "1")
			{
				//delete link
				if(isset($link->directory_company_category_id))
				{
					$saved = $link->delete() && $saved;
				}
			}
		}

		if($saved)
		{
			$this->cache->clean();
			$this->messages[] = "Company Saved";
		}
		else
		{
			$this->errors[] = "Error Saving Company";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->directory_company($company->company_id);
		}
		else
		{
			$this->directory_companies();
		}
	}

	public function directory_company_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_COMPANIES))
		{
			return $this->denied($this->data);
		}

		$this->generic_delete("directory_company", "directory_companies");
	}
	/* end directory company functions */

	public function directory_export()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DIRECTORY_COMPANIES))
		{
			return $this->denied($this->data);
		}

		$this->load->dbutil();

		$result = $this->db->query("SELECT c.*, cat.category_name AS Category
		                           FROM directory_companies AS c
		                           INNER JOIN directory_companies_categories AS link ON c.company_id=link.company_id
		                           INNER JOIN directory_categories AS cat ON cat.category_id=link.category_id
		                           WHERE c.site_id=?
		                           ORDER BY published DESC, company, category_name",
		                           array(Mainframe::active_site_id()));

		header('Content-Type: text/csv');
		header('Content-Disposition: attachment; filename="Directory Export ' . date("Y-m-d H:i") . '.csv"');
		header('Pragma: no-cache');
		echo $this->dbutil->csv_from_result($result);
	}

	public function contact_responses_export()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES))
		{
			return $this->denied($this->data);
		}

		$this->load->dbutil();

		$result = $this->db->query("SELECT *
									FROM `contact_responses`
									WHERE `site_id`=? AND `module_instance_id`=? AND `timestamp` >= ? AND `timestamp` <= ?
									ORDER BY `timestamp`",
									array(
										Mainframe::active_site_id(),
										$this->input->post("module_instance_id"),
										$this->input->post("date_start"),
										$this->input->post("date_end"),
									));

		header('Content-Type: text/csv');
		header('Content-Disposition: attachment; filename="Contact Form Responses ' . date("Y-m-d H:i") . '.csv"');
		header('Pragma: no-cache');
		echo $this->dbutil->csv_from_result($result);
	}

	public function crawler_issues()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/crawler_issues", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function maintenance()
	{
		$this->admin_login_required();

		if(!$this->current_user->has(ACTION_CONFIG))
		{
			return $this->denied($this->data);
		}

		set_time_limit(600);

		// Find any long running queries and kill them.
		$query 				= $this->db->query("show processlist");
		$mysql_killed 		= 0;
		$tables_defragged 	= 0;

		foreach($query->result() as $process)
		{
			if($process->Time > 500)
			{
				$this->db->query("KILL $process->Id");
				$mysql_killed++;
			}
		}

		if($mysql_killed > 0)
		{
			$this->messages[] = $mysql_killed . " MySQL queries killed.";
		}

		// Optimize database.
		$this->load->dbutil();
		$this->dbutil->optimize_database();
		$this->messages[] = "Database Optimized";

		$this->index();
	}

	function upload_image()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_MODULES))
		{
			return $this->denied($this->data);
		}

		$config['upload_path'] 		= ABSOLUTE_PATH . '/images/';
		$config['allowed_types'] 	= 'gif|jpg|jpeg|png';
		$config['max_size']			= '5242880';	//5MB
		$config['encrypt_name']		= false;
		$config['remove_spaces']	= true;
		$config['overwrite'] 		= false;

		$this->load->library('upload', $config);
		$filename = null;

		if($this->upload->do_upload("image"))
		{
			$upload_data = $this->upload->data();
			$filename = "/images/" . $upload_data["file_name"];

			// Return the size of the uploaded image.
			// If it appears to be a retina image, return half the size.
			$size 	= @getimagesize(ABSOLUTE_PATH . $filename);
			$width 	= $size[0];
			$height = $size[1];

			if(preg_match('/@2x\./', $filename))
			{
				$width 	/= 2;
				$height /= 2;
			}

			http_response_code(200);
			$this->spit_json(array("src" => $filename, "width" => $width, "height" => $height));
		}
		else
		{
			$errors = implode('\n', $this->upload->error_msg);
			http_response_code(409);
			die($errors);
		}
	}

	public function clean_cache()
	{
		$this->init();
		$this->clean_css_cache();
		$this->clean_application_cache();

		$this->messages[] = "CSS cache was cleaned.";
		$this->messages[] = "Application cache was cleaned.";
		$this->index();
	}

	public function changelog()
	{
		$this->init();
		require_once(APPLICATION_PATH . "/third_party/Michelf/Markdown.inc.php");

		$this->data["changelog"] = Markdown::defaultTransform(file_get_contents(ABSOLUTE_PATH . "/CHANGELOG.md"));

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("changelog", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function errorlog()
	{
		$this->init();

		$errorlog = false;

		if(file_exists(ABSOLUTE_PATH . "/error_log"))
		{
			$errorlog = ABSOLUTE_PATH . "/error_log";
		}
		else if(file_exists(APPLICATION_PATH . "/logs/error_log"))
		{
			$errorlog = APPLICATION_PATH . "/logs/error_log";
		}

		if($errorlog)
		{
			$errorlog = nl2br(file_get_contents($errorlog));
		}

		$this->data["errorlog"] = $errorlog;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("errorlog", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function errorlog_delete()
	{
		$this->init();

		$errorlog = false;

		if(file_exists(ABSOLUTE_PATH . "/error_log"))
		{
			$errorlog = ABSOLUTE_PATH . "/error_log";
		}
		else if(file_exists(APPLICATION_PATH . "/logs/error_log"))
		{
			$errorlog = APPLICATION_PATH . "/logs/error_log";
		}

		if($errorlog)
		{
			if(unlink($errorlog))
			{
				$this->messages = "Error Log Deleted";
			}
			else
			{
				$this->errors[] = "Could Not Delete Error Log";
			}
		}
		else
		{
			$this->errors[] = "No Error Log Found";
		}

		$this->errorlog();
	}

	public function seo($sitemap=0)
	{
		$this->init();
		$this->load->model("template");

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$ga_code = Mainframe::site()->ga_code;

		if(!$ga_code)
		{
			$t = new Template();
			$t->load(Mainframe::site()->template_id);

			$template_code = file_get_contents(ABSOLUTE_PATH . "/templates/" . $t->directory . "/index.php");
			preg_match('/(UA-........-.)/', $template_code, $matches);

			if(count($matches) >= 1)
			{
				$ga_code = $matches[0];
			}
		}

		$this->data["ga_code"] = $ga_code;

		if($sitemap == 1)
		{
			$this->messages[] = 'A sitemap was saved to <a href="' . LIVE_SITE . '/sitemap.xml" target="_blank">' . LIVE_SITE . '/sitemap.xml</a>';
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/seo", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	/* redirect functions */
	public function redirects()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("redirect", "redirects");
	}

	public function redirect($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model("redirect");

		$redirect = new Redirect();
		$redirect->load($id);

		if($this->input->get("old") != "")
		{
			$redirect->old_url = $this->input->get("old");
		}

		$pages = $this->page->get();
		$site_id = ($redirect->site_id ? $redirect->site_id : Mainframe::active_site_id());

		$this->data["id"] 		= $id;
		$this->data["return"] 	= ($this->input->get("return") ? $this->input->get("return") : "/admin/redirects");
		$this->data["redirect"] = $redirect;
		$this->data["pages"] 	= $pages;
		$this->data["site_id"] 	= $site_id;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/redirect_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function redirect_save()
	{
		$this->init();
		csrf_verify();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->model("redirect");

		$id = $this->input->post("redirect_id");

		$redirect = new redirect();
		$redirect->load($id);
		$redirect->readPostVars();

		// Remove leading slash if present.
		if(startsWith($redirect->old_url, "/"))
		{
			$redirect->old_url = substr($redirect->old_url, 1);
		}

		if($redirect->save())
		{
			$this->cache->clean();
			$this->clean_css_cache();
			$this->messages[] = "URL Redirect Saved";
		}
		else
		{
			$this->errors[] = "Error Saving URL Redirect";
		}

		if($this->input->post("submit_action") == "continue")
		{
			$this->redirect($redirect->redirect_id);
		}
		else
		{
			if($this->input->post("return") == "/admin/http404s")
			{
				$this->http404s();
			}
			else
			{
				$this->redirects();
			}
		}
	}

	public function redirect_delete()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->clean_css_cache();
		$this->generic_delete("redirect", "redirects");
	}
	/* end redirect functions */

	public function http404s()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_SEO))
		{
			return $this->denied($this->data);
		}

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/404_list", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	/* document functions */
	public function documents()
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DOCUMENTS))
		{
			return $this->denied($this->data);
		}

		$this->generic_list("document", "documents");
	}

	public function document($id=0)
	{
		$this->init();

		if(!$this->current_user->has(ACTION_DOCUMENTS) ||
		   !$this->acl->checkDocument($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("category", "document"));

		$document = new document();
		$document->load($id);

		$sites 			= $this->site->get();
		$categories 	= $this->category->get();
		$site_id 		= ($document->site_id ? $document->site_id : Mainframe::active_site_id());
		$files 			= $this->document->getFiles($site_id, $this->current_user);
		$acls 			= $this->acl->loadByDocumentID($document->document_id);

		$this->data["id"] 			= $id;
		$this->data["document"] 	= $document;
		$this->data["files"] 		= $files;
		$this->data["site_id"] 		= $site_id;
		$this->data["sites"] 		= $sites;
		$this->data["categories"]	= $categories;
		$this->data["acls"]			= $acls;

		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("admin/document_edit", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	public function document_save($ajax=false)
	{
		$this->init();
		$id = $this->input->post("document_id");

		if(!$this->current_user->has(ACTION_DOCUMENTS) ||
		   !$this->acl->checkDocument($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("document", "redirect", "document_log", "document_tag"));

		$document 		= new document();
		$document->load($id);
		$old_document 	= clone($document);
		$old_url 		= $document->url;
		$old_filename 	= $document->filename;

		if($ajax)
		{
			$document->site_id 		= $this->input->post("site_id");
			$document->category_id 	= ($this->input->post("category_id") ? $this->input->post("category_id") : null);
			$document->title 		= $this->input->post("title");
			$document->description 	= ($this->input->post("description") ? $this->input->post("description") : null);
			$document->filename 	= $this->input->post("filename");
			$document->url 			= $this->input->post("url");
			$document->access 		= $this->input->post("access");
			$document->published 	= $this->input->post("published");
		}
		else
		{
			$document->readPostVars();
		}

		if(!$id)
		{
			$document->created = date("Y-m-d H:i:s");
			$document->created_user_id = $this->current_user->user_id;
		}
		else
		{
			$document->modified = date("Y-m-d H:i:s");
			$document->modified_user_id = $this->current_user->user_id;
		}

		// Check if this URL is available, add numbers on the end if it isn't.
		$count 	= 0;
		$url 	= $document->url;
		do
		{
			$query = $this->db->query("SELECT `document_id` FROM `document` WHERE `document_id` !=? AND `url`=?", array($document->document_id, $url));

			if($query->num_rows() == 0)
			{
				break;
			}

			$count++;
			$url = $document->url . $count;
		}
		while(true);

		$document->url 	= $url;
		$new_url 		= $document->url;

		if($this->input->post("filename") == "0" && !isset($_FILES["upload_file"]))
		{
			if($ajax)
			{
				http_response_code(409);
				die("You must upload a file from your computer or select an existing file.");
			}
			else
			{
				$this->errors[] = "You must upload a file from your computer or select an existing file.";
				$this->document($document->document_id);
				return;
			}
		}

		$config = array();
		$config['allowed_types']	= 'jpg|png|gif|txt|md|csv|tsv|pdf|doc|docx|xls|xlsx|zip|gz|7z|odt|ods|ai|svg|mp3|wav|mov|mp4|mpeg|mpg|log|ppt|pptx|ppsx';
		$config['upload_path'] 		= DOCUMENT_PATH . '/';
		$config['file_ext_tolower'] = true;
		$config['overwrite'] 		= ($this->input->post("overwrite") == 1 ? true : false);
		$this->load->library('upload', $config);

		if(isset($_FILES["upload_file"]) && $_FILES["upload_file"]["name"] != "")
		{
			if($this->upload->do_upload("upload_file"))
			{
				$upload_data 				= $this->upload->data();
				$document->filename 		= $upload_data["file_name"];

				// If there was an old filename, check if any other files reference it.
				if($old_filename)
				{
					$morefiles = $this->document->loadByFilename($old_filename, $this->current_user);

					if(count($morefiles) == 0)
					{
						unlink(DOCUMENT_PATH . "/" . $old_filename);
					}
				}
			}
			else
			{
				if($ajax)
				{
					http_response_code(409);
					die(implode('\n', $this->upload->error_msg));
				}
				else
				{
					if(is_array($this->upload->error_msg))
					{
						$this->errors = array_merge($this->errors, $this->upload->error_msg);
					}
					else
					{
						$this->errors[] = $this->upload->error_msg;
					}

					$this->document();
					return;
				}
			}
		}

		if($document->save())
		{
			$this->cache->clean();

			// Tags
			$ids = "";
			$tags = explode(",", $this->input->post("tag_ids"));

			foreach($tags as $t)
			{
				$tag = new Document_tag();

				if(is_numeric($t))
				{
					$tag->load($t);
				}
				else if(trim($t) != "")
				{
					$t = trim($t);
					$tag->loadByTagName($t);
					$tag->tag = $t;
					$tag->save();
				}

				if($tag->tag_id)
				{
					$this->document_tag->add_link($tag->tag, $document->document_id);
					$ids .= $tag->tag_id . ",";
				}
			}

			if($ids)
			{
				$ids = substr($ids, 0, -1);
				$this->document_tag->deleteAllOtherTags($document->document_id, $ids);
			}
			else
			{
				$this->document_tag->deleteAllTags($document->document_id);
			}

			// Save ACL
			$roles = $this->acl_role->get();

			foreach($roles as $role)
			{
				$acl = new ACL();
				$acl->loadByDocumentRoleID($document->document_id, $role->acl_role_id);

				if(!$acl->acl_id)
				{
					$acl->document_id 	= $document->document_id;
					$acl->acl_role_id 	= $role->acl_role_id;
				}

				if($this->input->post("acl_" . $role->acl_role_id) == "r")
				{
					$acl->read = 1;
					$acl->write = 0;
				}
				else if($this->input->post("acl_" . $role->acl_role_id) == "rw")
				{
					$acl->read = 1;
					$acl->write = 1;
				}
				else
				{
					$acl->read = 0;
					$acl->write = 0;
				}
				$acl->save();
			}

			$this->document_log->logChanges($old_document, $document, $this->current_user);

			// If the URL has been updated, we'll automatically add a redirect for the old URL.
			if($old_url && $old_url != $new_url)
			{
				$redirect = new Redirect();
				$redirect->loadByURL($old_url, $document->site_id);
				$redirect->old_url = "document/" . $old_url;
				$redirect->new_url = "document/" . $new_url;
				$redirect->site_id = $document->site_id;
				$redirect->page_id = null;
				$redirect->save();

				$this->messages[] = "A redirect was automatically created because you changed the page's URL.";
			}

			$this->cache->clean();

			if($ajax)
			{
				http_response_code(200);
				$this->spit_json($document);
			}
			else
			{
				$this->messages[] = "Document Saved";
			}
		}
		else
		{
			if($ajax)
			{
				http_response_code(409);
				die("Error Saving Document");
			}
			else
			{
				$this->errors[] = "Error Saving Document";
			}
		}

		if(!$ajax)
		{
			if($this->input->post("submit_action") == "continue")
			{
				$this->document($document->document_id);
			}
			else
			{
				$this->documents();
			}
		}
	}

	public function document_delete($ajax=false)
	{
		$this->init();
		$id = $this->input->get_post("document_id");

		if(!$this->current_user->has(ACTION_DOCUMENTS) ||
		   !$this->acl->checkDocument($id, $this->current_user, "write"))
		{
			return $this->denied($this->data);
		}

		$this->load->model(array("document", "document_log"));

		$doc 	= new Document();
		$doc->load($id);
		$old_document = clone($doc);
		$old_filename = $doc->filename;

		if($doc->url != "")
		{
			$check1 = $this->db->query("SELECT COUNT(*) AS total, GROUP_CONCAT(p.`title` SEPARATOR '</li><li>') AS pages
			                           FROM `content_values` AS v
			                           INNER JOIN `pages` AS p USING(page_id)
			                           WHERE v.`value` LIKE '%document/" . $this->db->escape_str($doc->url) . "%'");
			$check2 = $this->db->query("SELECT COUNT(*) AS total, GROUP_CONCAT(i.`tag` SEPARATOR '</li><li>') AS modules
			                           FROM `module_settings` AS s
			                           INNER JOIN `module_instances` AS i USING(module_instance_id)
			                           WHERE s.`value` LIKE '%document/" . $this->db->escape_str($doc->url) . "%'");
			$row1 = $check1->row();
			$row2 = $check2->row();
			$total1 = $row1->total;
			$total2 = $row2->total;
			$total_links = $total1 + $total2;
			$link_details = "";

			if($total_links > 0)
			{
				if($total1 > 0)
				{
					$link_details .= "<br /><br />Pages with links:<ul><li>" . $row1->pages . "</li></ul>";
				}
				if($total2 > 0)
				{
					$link_details .= "<br /><br />Modules with links:<ul><li>" . $row2->modules . "</li></ul>";
				}

				if($ajax)
				{
					http_response_code(409);
					die("You can not delete this document because " . $total_links . " active link" . ($total_links == 1 ? " exists" : "s exist") . " on your website." . $link_details);
				}
				else
				{
					$this->errors[] = "You can not delete this document because " . $total_links . " active link" . ($total_links == 1 ? " exists" : "s exist") . " on your website." . $link_details;
					$this->documents();
					return;
				}
			}
		}

		if($doc->delete())
		{
			$this->document_log->logDeleted($old_document, $this->current_user);

			// Check if any other files reference it.
			$morefiles = $this->document->loadByFilename($old_filename, $this->current_user);

			if(count($morefiles) == 0)
			{
				unlink(DOCUMENT_PATH . "/" . $old_filename);
			}

			if($ajax)
			{
				http_response_code(200);
			}
			else
			{
				$this->messages[] = "Document Deleted";
			}
		}
		else
		{
			if($ajax)
			{
				http_response_code(409);
				die("Error Deleting Document");
			}
			else
			{
				$this->errors[] = "Error Deleting Document";
			}
		}


		if(!$ajax)
		{
			$this->documents();
		}
	}
	/* end document functions */

	public function qatest()
	{
		$this->init();

		if(ENVIRONMENT == "development")
		{
			$this->load->library('unit_test');
			$this->load->model(array("shop/Shop_cart", "shop/Shop_category_layout", "shop/Shop_category_product", "shop/Shop_category_shopper_group", "shop/Shop_category", "shop/Shop_config", "shop/Shop_currency", "shop/Shop_datasource", "shop/Shop_image", "shop/Shop_layout_specific_data", "shop/Shop_order_status", "shop/Shop_order", "shop/Shop_payment_gateway", "shop/Shop_postalcode", "shop/Shop_price", "shop/Shop_product_layout", "shop/Shop_product_option", "shop/Shop_product_related", "shop/Shop_product", "shop/Shop_route", "shop/Shop_shipping_gateway", "shop/Shop_shopper_group", "shop/Shop_tax", "shop/Shop_vehicle", "shop/Shop_product_variant", "Acl_action", "Acl_role_action", "Acl_role", "Acl_user_role", "Acl", "Appointment", "Category", "Content_type", "Content_value", "Crawler_issue", "Directory_category_link", "Directory_category", "Directory_company", "Directory_coupon", "Document_log", "Document", "Domain", "Event_form", "Event", "Four_oh_four", "Menu", "Menuitem", "Metadata_page", "Module", "Moduleinstance", "Modulepagelink", "Modulesetting", "Page", "Redirect", "Site", "Template", "User", "page_tag"));

			error_reporting(-1);
			ini_set('display_errors', 1);

			$this->db->query("DELETE FROM `shop_products`");
			$this->db->query("ALTER TABLE `shop_products` AUTO_INCREMENT 1");
			$p = new Shop_product();
			$p->load(1);
			$p->product_name = "Dummy Product";
			$p->product_layout_id = 1;
			$p->url = "p";
			$p->save();

			$this->db->query("DELETE FROM `shop_product_variant`");
			$this->db->query("ALTER TABLE `shop_product_variant` AUTO_INCREMENT 1");
			$v = new Shop_product_variant();
			$v->load(1);
			$v->product_id = 1;
			$v->variant = "Dummy Variant";
			$v->sort = 1;
			$v->save();

			$this->db->query("DELETE FROM `shop_vehicles`");
			$this->db->query("ALTER TABLE `shop_vehicles` AUTO_INCREMENT 1");
			$v = new Shop_vehicle();
			$v->load(1);
			$v->year = 2016;
			$v->make = "Toyota";
			$v->model = "Matrix";
			$v->style = "x";
			$v->save();

			$this->db->query("DELETE FROM `shop_categories`");
			$this->db->query("ALTER TABLE `shop_categories` AUTO_INCREMENT 1");
			$c = new Shop_category();
			$c->load(1);
			$c->category_name = "Dummy Category";
			$c->category_layout_id = 1;
			$c->default_product_layout_id = 1;
			$c->url = "c";
			$c->save();

			$this->db->query("DELETE FROM `document`");
			$this->db->query("ALTER TABLE `document` AUTO_INCREMENT 1");
			$d = new Document();
			$d->load(1);
			$d->site_id = 1;
			$d->save();

			$this->db->query("DELETE FROM `pages`");
			$this->db->query("ALTER TABLE `pages` AUTO_INCREMENT 1");
			$pg = new Page();
			$pg->load(1);
			$pg->site_id = 1;
			$pg->url = "pg";
			$pg->save();

			$this->db->query("DELETE FROM `shop_layout_specific_data`");
			$this->db->query("ALTER TABLE `shop_layout_specific_data` AUTO_INCREMENT 1");
			$x = new Shop_Layout_Specific_Data();
			$x->load(1);
			$x->data_name = "Dummy Data";
			$x->key = "x";
			$x->save();

			$this->db->query("DELETE FROM `shop_routes`");
			$this->db->query("ALTER TABLE `shop_routes` AUTO_INCREMENT 1");

			$this->unit->set_test_items(array('test_name', 'result', 'notes'));
			$str = '
	{rows}
	        <tr>
	                <td>{item}</td>
	                <td>{result}</td>
	        </tr>
	{/rows}
	        <tr><td colspan="2">&nbsp;</td></tr>
	';

			$this->unit->set_template($str);

			$this->unit->run(new Acl(), "is_object", "Model: Acl->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Acl->load(1), "is_bool", "Model: Acl->Load()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByPageID(1), "is_array", "Model: Acl->loadByPageID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByModuleInstanceID(1), "is_array", "Model: Acl->loadByModuleInstanceID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByDocumentID(1), "is_array", "Model: Acl->loadByDocumentID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByMenuItemID(1), "is_array", "Model: Acl->loadByMenuItemID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByPageRoleID(1, 3), "is_bool", "Model: Acl->loadByPageRoleID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByModuleRoleID(1, 3), "is_bool", "Model: Acl->loadByModuleRoleID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByDocumentRoleID(1, 3), "is_bool", "Model: Acl->loadByDocumentRoleID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->loadByMenuItemRoleID(1, 3), "is_bool", "Model: Acl->loadByMenuItemRoleID()", "Checking for correct data type.");
			$this->unit->run($this->Acl->checkPage(1, $this->current_user, "read"), "is_bool", "Model: Acl->checkPage()", "Checking for correct data type.");
			$this->unit->run($this->Acl->checkModule(1, $this->current_user, "read"), "is_bool", "Model: Acl->checkModule()", "Checking for correct data type.");
			$this->unit->run($this->Acl->checkDocument(1, $this->current_user, "read"), "is_bool", "Model: Acl->checkDocument()", "Checking for correct data type.");
			$this->unit->run($this->Acl->checkMenuItem(1, $this->current_user, "read"), "is_bool", "Model: Acl->checkMenuItem()", "Checking for correct data type.");

			$this->unit->run(new Acl_action(), "is_object", "Model: Acl_action->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Acl_action->load(1), "is_bool", "Model: Acl_action->Load()", "Checking for correct data type.");
			$this->unit->run($this->Acl_action->loadByRoleID(3), "is_array", "Model: Acl_action->loadByRoleID()", "Checking for correct data type.");
			$this->unit->run($this->Acl_action->getConstants(), "is_array", "Model: Acl_action->getConstants()", "Checking for correct data type.");

			$this->unit->run(new Acl_role_action(), "is_object", "Model: Acl_role_action->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Acl_role_action->load(1), "is_bool", "Model: Acl_role_action->Load()", "Checking for correct data type.");
			$this->unit->run($this->Acl_role_action->add_link(3, 1), "is_bool", "Model: Acl_role_action->add_link()", "Checking for correct data type.");
			$this->unit->run($this->Acl_role_action->delete_link(3, 1), "is_bool", "Model: Acl_role_action->delete_link()", "Checking for correct data type.");

			$this->unit->run(new Acl_role(), "is_object", "Model: Acl_role->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Acl_role->load(1), "is_bool", "Model: Acl_role->Load()", "Checking for correct data type.");
			$this->unit->run($this->Acl_role->loadByUserID(1), "is_array", "Model: Acl_role->loadByUserID()", "Checking for correct data type.");

			$this->unit->run(new Acl_user_role(), "is_object", "Model: Acl_user_role->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Acl_user_role->load(1), "is_bool", "Model: Acl_user_role->Load()", "Checking for correct data type.");
			$this->unit->run($this->Acl_user_role->add_link(1, 1), "is_bool", "Model: Acl_user_role->add_link()", "Checking for correct data type.");
			$this->unit->run($this->Acl_user_role->delete_link(1, 1), "is_bool", "Model: Acl_user_role->delete_link()", "Checking for correct data type.");

			$this->unit->run(new Appointment(), "is_object", "Model: Appointment->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Appointment->load(1), "is_bool", "Model: Appointment->Load()", "Checking for correct data type.");

			$this->unit->run(new Category(), "is_object", "Model: Category->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Category->load(1), "is_bool", "Model: Category->Load()", "Checking for correct data type.");
			$this->unit->run($this->Category->LoadBySiteID(1), "is_array", "Model: Category->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Category->get(), "is_array", "Model: Category->get()", "Checking for correct data type.");
			$this->unit->run($this->Category->loadDocumentCategories(1), "is_array", "Model: Category->loadDocumentCategories()", "Checking for correct data type.");

			$this->unit->run(new Content_type(), "is_object", "Model: Content_type->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Content_type->load(1), "is_bool", "Model: Content_type->Load()", "Checking for correct data type.");
			$this->unit->run($this->Content_type->get(), "is_array", "Model: Content_type->get()", "Checking for correct data type.");

			$this->unit->run(new Content_value(), "is_object", "Model: Content_value->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Content_value->load(1), "is_bool", "Model: Content_value->Load()", "Checking for correct data type.");
			$this->unit->run($this->Content_value->LoadByPageKey(1, "content"), "is_bool", "Model: Content_value->LoadByPageKey()", "Checking for correct data type.");
			$this->unit->run($this->Content_value->LoadByPageID(1), "is_array", "Model: Content_value->LoadByPageID()", "Checking for correct data type.");
			$this->unit->run($this->Content_value->delete_all(1), "is_null", "Model: Content_value->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Content_value->delete_all_others(1, "2,3"), "is_null", "Model: Content_value->delete_all_others()", "Checking for correct data type.");

			$this->unit->run(new Crawler_issue(), "is_object", "Model: Crawler_issue->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Crawler_issue->load(1), "is_bool", "Model: Crawler_issue->Load()", "Checking for correct data type.");
			$this->unit->run($this->Crawler_issue->get(), "is_array", "Model: Crawler_issue->get()", "Checking for correct data type.");
			$this->unit->run($this->Crawler_issue->reset(), "is_null", "Model: Crawler_issue->reset()", "Checking for correct data type.");

			$this->unit->run(new Directory_category_link(), "is_object", "Model: Directory_category_link->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category_link->load(1), "is_bool", "Model: Directory_category_link->Load()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category_link->LoadByCompanyCategoryIDs(1,1), "is_bool", "Model: Directory_category_link->LoadByCompanyCategoryIDs()", "Checking for correct data type.");

			$this->unit->run(new Directory_category(), "is_object", "Model: Directory_category->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category->load(1), "is_bool", "Model: Directory_category->Load()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category->LoadBySiteID(1), "is_array", "Model: Directory_category->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category->LoadByParentID(1), "is_array", "Model: Directory_category->LoadByParentID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category->LoadByCompanyID(1), "is_array", "Model: Directory_category->LoadByCompanyID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_category->loadByURL(1, "x"), "is_bool", "Model: Directory_category->loadByURL()", "Checking for correct data type.");

			$this->unit->run(new Directory_company(), "is_object", "Model: Directory_company->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Directory_company->load(1), "is_bool", "Model: Directory_company->Load()", "Checking for correct data type.");
			$this->unit->run($this->Directory_company->LoadBySiteID(1), "is_array", "Model: Directory_company->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_company->LoadByCategoryID(1, 1), "is_array", "Model: Directory_company->LoadByCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_company->LoadByKeyword("x", 1), "is_array", "Model: Directory_company->LoadByKeyword()", "Checking for correct data type.");
			$this->unit->run($this->Directory_company->loadByURL(1, "x"), "is_bool", "Model: Directory_company->loadByURL()", "Checking for correct data type.");

			$this->unit->run(new Directory_coupon(), "is_object", "Model: Directory_coupon->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Directory_coupon->load(1), "is_bool", "Model: Directory_coupon->Load()", "Checking for correct data type.");
			$this->unit->run($this->Directory_coupon->LoadBySiteID(1), "is_array", "Model: Directory_coupon->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Directory_coupon->LoadByCompanyID(1), "is_array", "Model: Directory_coupon->LoadByCompanyID()", "Checking for correct data type.");

			$this->unit->run(new Document_log(), "is_object", "Model: Document_log->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Document_log->load(1), "is_bool", "Model: Document_log->Load()", "Checking for correct data type.");
			$this->unit->run($this->Document_log->logChanges(new Document(), new Document(), $this->current_user), "is_bool", "Model: Document_log->logChanges()", "Checking for correct data type.");
			$this->unit->run($this->Document_log->logDeleted(new Document(), $this->current_user), "is_bool", "Model: Document_log->logDeleted()", "Checking for correct data type.");

			$this->unit->run(new Document(), "is_object", "Model: Document->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Document->load(1), "is_bool", "Model: Document->Load()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadBySiteID(1, $this->current_user), "is_array", "Model: Document->loadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadByCategoryID(1, 1, null, "OR", "created", "x", $this->current_user), "is_array", "Model: Document->loadByCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadByCategoryIDYear(1, 1, null, "OR", 2017, "created", "x", $this->current_user), "is_array", "Model: Document->loadByCategoryIDYear()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadByURL("x"), "is_bool", "Model: Document->loadByURL()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadByDirectory(1, "/", $this->current_user), "is_array", "Model: Document->loadByDirectory()", "Checking for correct data type.");
			$this->unit->run($this->Document->getFiles(1, $this->current_user), "is_array", "Model: Document->getFiles()", "Checking for correct data type.");
			$this->unit->run($this->Document->loadByFilename("x", $this->current_user), "is_array", "Model: Document->loadByFilename()", "Checking for correct data type.");
			$this->unit->run($this->Document->search("x", $this->current_user), "is_array", "Model: Document->search()", "Checking for correct data type.");

			$this->unit->run(new Domain(), "is_object", "Model: Domain->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Domain->load(1), "is_bool", "Model: Domain->Load()", "Checking for correct data type.");
			$this->unit->run($this->Domain->LoadByDomain("x"), "is_bool", "Model: Domain->LoadByDomain()", "Checking for correct data type.");

			$this->unit->run(new Event_form(), "is_object", "Model: Event_form->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Event_form->load(1), "is_bool", "Model: Event_form->Load()", "Checking for correct data type.");
			$this->unit->run($this->Event_form->loadBySiteID(1), "is_array", "Model: Event_form->loadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Event_form->LoadActive(), "is_array", "Model: Event_form->LoadActive()", "Checking for correct data type.");
			$this->unit->run($this->Event_form->LoadByEventID(2, 1), "is_array", "Model: Event_form->LoadByEventID()", "Checking for correct data type.");
			$this->unit->run($this->Event_form->get(), "is_array", "Model: Event_form->get()", "Checking for correct data type.");

			$this->unit->run(new Event(), "is_object", "Model: Event->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Event->load(1), "is_bool", "Model: Event->Load()", "Checking for correct data type.");
			$this->unit->run($this->Event->LoadWithDetails(1), "is_bool", "Model: Event->LoadWithDetails()", "Checking for correct data type.");
			$this->unit->run($this->Event->LoadBySiteID(1), "is_array", "Model: Event->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Event->LoadParents(1), "is_array", "Model: Event->LoadParents()", "Checking for correct data type.");
			$this->unit->run($this->Event->LoadByParentID(1), "is_array", "Model: Event->LoadByParentID()", "Checking for correct data type.");
			$this->unit->run($this->Event->LoadEventStream(1), "is_array", "Model: Event->LoadEventStream()", "Checking for correct data type.");
			$this->unit->run($this->Event->loadByURL(1, "x"), "is_bool", "Model: Event->loadByURL()", "Checking for correct data type.");
			$this->unit->run($this->Event->get(), "is_array", "Model: Event->get()", "Checking for correct data type.");
			$this->unit->run($this->Event->search("x"), "is_array", "Model: Event->search()", "Checking for correct data type.");

			$this->unit->run(new Four_oh_four(), "is_object", "Model: Four_oh_four->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Four_oh_four->load(1), "is_bool", "Model: Four_oh_four->Load()", "Checking for correct data type.");
			$this->unit->run($this->Four_oh_four->loadByURL("x"), "is_bool", "Model: Four_oh_four->loadByURL()", "Checking for correct data type.");

			$this->unit->run(new Menu(), "is_object", "Model: Menu->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Menu->load(1), "is_bool", "Model: Menu->Load()", "Checking for correct data type.");
			$this->unit->run($this->Menu->LoadBySiteID(1), "is_array", "Model: Menu->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Menu->build_tree($this->current_user), "is_object", "Model: Menu->build_tree()", "Checking for correct data type.");

			$this->unit->run(new Menuitem(), "is_object", "Model: Menuitem->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Menuitem->load(1), "is_bool", "Model: Menuitem->Load()", "Checking for correct data type.");
			$this->unit->run($this->Menuitem->LoadByMenuID(1, null, $this->current_user), "is_array", "Model: Menuitem->LoadByMenuID()", "Checking for correct data type.");
			$this->unit->run($this->Menuitem->delete_all(1), "is_null", "Model: Menuitem->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Menuitem->delete_others(1, "2,3"), "is_null", "Model: Menuitem->delete_others()", "Checking for correct data type.");
			$this->unit->run($this->Menuitem->search("x", $this->current_user), "is_array", "Model: Menuitem->search()", "Checking for correct data type.");

			$this->unit->run(new Metadata_page(), "is_object", "Model: Metadata_page->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Metadata_page->load(1), "is_bool", "Model: Metadata_page->Load()", "Checking for correct data type.");
			$this->unit->run($this->Metadata_page->LoadByURL(1, "x"), "is_bool", "Model: Metadata_page->LoadByURL()", "Checking for correct data type.");
			$this->unit->run($this->Metadata_page->LoadBySiteID(1), "is_array", "Model: Metadata_page->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Metadata_page->LoadAllPagesBySiteID(1), "is_array", "Model: Metadata_page->LoadAllPagesBySiteID()", "Checking for correct data type.");

			$this->unit->run(new Module(), "is_object", "Model: Module->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Module->load(1), "is_bool", "Model: Module->Load()", "Checking for correct data type.");
			$this->unit->run($this->Module->search("x"), "is_array", "Model: Module->search()", "Checking for correct data type.");

			$this->unit->run(new Moduleinstance(), "is_object", "Model: Moduleinstance->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->load(1), "is_bool", "Model: Moduleinstance->Load()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->install(), "is_bool", "Model: Moduleinstance->install()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->module_settings(), "is_array", "Model: Moduleinstance->module_settings()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->scripts(), "is_array", "Model: Moduleinstance->scripts()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->enable_jquery(), "is_bool", "Model: Moduleinstance->enable_jquery()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->enable_jquery_ui(), "is_bool", "Model: Moduleinstance->enable_jquery_ui()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->children(), "is_array", "Model: Moduleinstance->children()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->load(1), "is_bool", "Model: Moduleinstance->load()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->load_settings(), "is_null", "Model: Moduleinstance->load_settings()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->setting("x"), "is_string", "Model: Moduleinstance->setting()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->set("x", "y"), "is_null", "Model: Moduleinstance->set()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->LoadByPageID(1, $this->current_user), "is_array", "Model: Moduleinstance->LoadByPageID()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->LoadBySiteID(1, "tag", $this->current_user), "is_array", "Model: Moduleinstance->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->LoadInstalledByPageID(1, 1), "is_array", "Model: Moduleinstance->LoadInstalledByPageID()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->loadDefaultModules(1), "is_array", "Model: Moduleinstance->loadDefaultModules()", "Checking for correct data type.");
			$this->unit->run($this->Moduleinstance->search("x", $this->current_user), "is_array", "Model: Moduleinstance->search()", "Checking for correct data type.");

			$this->unit->run(new Modulepagelink(), "is_object", "Model: Modulepagelink->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Modulepagelink->load(1), "is_bool", "Model: Modulepagelink->Load()", "Checking for correct data type.");
			$this->unit->run($this->Modulepagelink->LoadByModulePageIDs(1, 1), "is_bool", "Model: Modulepagelink->LoadByModulePageIDs()", "Checking for correct data type.");
			$this->unit->run($this->Modulepagelink->add_link(1, 1), "is_bool", "Model: Modulepagelink->add_link()", "Checking for correct data type.");
			$this->unit->run($this->Modulepagelink->delete_all(1), "is_null", "Model: Modulepagelink->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Modulepagelink->delete_others(1, "2,3"), "is_null", "Model: Modulepagelink->delete_others()", "Checking for correct data type.");

			$this->unit->run(new Modulesetting(), "is_object", "Model: Modulesetting->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Modulesetting->load(1), "is_bool", "Model: Modulesetting->Load()", "Checking for correct data type.");
			$this->unit->run($this->Modulesetting->LoadByModuleKey(1, "x"), "is_bool", "Model: Modulesetting->LoadByModuleKey()", "Checking for correct data type.");
			$this->unit->run($this->Modulesetting->load_by_instance_id(1), "is_array", "Model: Modulesetting->load_by_instance_id()", "Checking for correct data type.");
			$this->unit->run($this->Modulesetting->delete_all(1), "is_null", "Model: Modulesetting->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Modulesetting->delete_all_others(1, "2,3"), "is_null", "Model: Modulesetting->delete_all_others()", "Checking for correct data type.");

			$this->unit->run(new Page(), "is_object", "Model: Page->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Page->load(1), "is_bool", "Model: Page->Load()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadBySitePageID(1, 1), "is_bool", "Model: Page->LoadBySitePageID()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadBySiteID(1, $this->current_user), "is_array", "Model: Page->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadBySiteModuleIDs(1, 1, $this->current_user), "is_array", "Model: Page->LoadBySiteModuleIDs()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadByCategoryContentTypeTagID(1, 1, 1, false, "OR", "`created` DESC", "x", $this->current_user, 0), "is_array", "Model: Page->LoadByCategoryContentTypeTagID()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadByCategoryContentTypeTagIDYear(1, 1, 1, 2017, false, "OR", "`created` DESC", "x", $this->current_user, 0), "is_array", "Model: Page->LoadByCategoryContentTypeTagIDYear()", "Checking for correct data type.");
			$this->unit->run($this->Page->LoadForRSSFeed(1), "is_array", "Model: Page->LoadForRSSFeed()", "Checking for correct data type.");
			$this->unit->run($this->Page->delete_all(1), "is_null", "Model: Page->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Page->delete_others(1, "2,3"), "is_null", "Model: Page->delete_others()", "Checking for correct data type.");
			$this->unit->run($this->Page->search("x", 1, $this->current_user), "is_array", "Model: Page->search()", "Checking for correct data type.");
			$this->unit->run($this->Page->field("x", "y"), "is_string", "Model: Page->field()", "Checking for correct data type.");
			$this->unit->run($this->Page->checkPublishedDates(), "is_null", "Model: Page->checkPublishedDates()", "Checking for correct data type.");
			$this->unit->run($this->Page->getUnlinked(1), "is_array", "Model: Page->getUnlinked()", "Checking for correct data type.");
			$this->unit->run($this->Page->getUnlinkedByMenuID(1, 1), "is_array", "Model: Page->getUnlinkedByMenuID()", "Checking for correct data type.");
			$this->unit->run($this->Page->url(1), "is_string", "Model: Page->url()", "Checking for correct data type.");

			$this->unit->run(new Page_Tag(), "is_object", "Model: Page->__construct()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->loadByPageID(1), "is_array", "Model: Page_tag->loadByPageID()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->loadByTagName("x"), "is_bool", "Model: Page_tag->loadByTagName()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->add_link("x", 1), "is_bool", "Model: Page_tag->add_link()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->delete_link("x", 1), "is_bool", "Model: Page_tag->delete_link()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->LoadTokens("x"), "is_array", "Model: Page_tag->LoadTokens()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->deleteAllOtherTags(1, "1,2"), "is_null", "Model: Page_tag->deleteAllOtherTags()", "Checking for correct data type.");
			$this->unit->run($this->page_tag->deleteAllTags(1), "is_null", "Model: Page_tag->deleteAllTags()", "Checking for correct data type.");

			$this->unit->run(new Redirect(), "is_object", "Model: Redirect->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Redirect->load(1), "is_bool", "Model: Redirect->Load()", "Checking for correct data type.");
			$this->unit->run($this->Redirect->loadByURL("x", 1), "is_bool", "Model: Redirect->loadByURL()", "Checking for correct data type.");

			$this->unit->run(new Site(), "is_object", "Model: Site->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Site->load(1), "is_bool", "Model: Site->Load()", "Checking for correct data type.");

			$this->unit->run(new Template(), "is_object", "Model: Template->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Template->load(1), "is_bool", "Model: Template->Load()", "Checking for correct data type.");

			$this->unit->run(new User(), "is_object", "Model: User->__construct()", "Checking for correct data type.");
			$this->unit->run($this->User->load(1), "is_bool", "Model: User->Load()", "Checking for correct data type.");
			$this->unit->run($this->User->LoadByUsername("x"), "is_bool", "Model: User->LoadByUsername()", "Checking for correct data type.");
			$this->unit->run($this->User->LoadByEmail("x"), "is_bool", "Model: User->LoadByEmail()", "Checking for correct data type.");
			$this->unit->run($this->User->authenticate("x", "x", true), "is_false", "Model: User->authenticate()", "Checking for correct data type.");
			$this->unit->run($this->current_user->has(ACTION_LOGIN), "is_bool", "Model: User->has()", "Checking for correct data type.");
			$this->unit->run($this->current_user->has_any(array(ACTION_LOGIN)), "is_bool", "Model: User->has_any()", "Checking for correct data type.");
			$this->unit->run($this->current_user->has_role(3), "is_bool", "Model: User->has_role()", "Checking for correct data type.");
			$this->unit->run($this->User->search("x"), "is_array", "Model: User->search()", "Checking for correct data type.");

			$this->unit->run(new Shop_cart(), "is_object", "Model: Shop_cart->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_cart->load(1), "is_bool", "Model: Shop_cart->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_cart->GetItemsBySessionID("asdf"), "is_array", "Model: Shop_cart->GetItemsBySessionID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_cart->AddItemByProductID(1, 1, "add"), "is_null", "Model: Shop_cart->AddItemByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_cart->ContainsItem(1), "is_bool", "Model: Shop_cart->ContainsItem()", "Checking for correct data type.");
			$this->unit->run($this->Shop_cart->EmptyBySessionID("asdf"), "is_null", "Model: Shop_cart->EmptyBySessionID()", "Checking for correct data type.");

			$this->unit->run(new Shop_category_layout(), "is_object", "Model: Shop_category_layout->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_layout->load(1), "is_bool", "Model: Shop_category_layout->Load()", "Checking for correct data type.");

			$this->unit->run(new Shop_category_product(), "is_object", "Model: Shop_category_product->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_product->load(1), "is_bool", "Model: Shop_category_product->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_product->add_link(1, 1), "is_null", "Model: Shop_category_product->add_link()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_product->delete_link(1, 1), "is_null", "Model: Shop_category_product->delete_link()", "Checking for correct data type.");

			$this->unit->run(new Shop_category_shopper_group(), "is_object", "Model: Shop_category_shopper_group->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_shopper_group->load(1), "is_bool", "Model: Shop_category_shopper_group->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_shopper_group->LoadBySiteIDWithLinks(1, 1), "is_array", "Model: Shop_category_shopper_group->LoadBySiteIDWithLinks()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_shopper_group->add_link(1, 1), "is_null", "Model: Shop_category_shopper_group->add_link()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_shopper_group->delete_link(1, 1), "is_null", "Model: Shop_category_shopper_group->delete_link()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category_shopper_group->validate_user($c, $this->current_user), "is_bool", "Model: Shop_category_shopper_group->validate_user()", "Checking for correct data type.");

			$this->unit->run(new Shop_category(), "is_object", "Model: Shop_category->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->load(1), "is_bool", "Model: Shop_category->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->GetURL(), "is_string", "Model: Shop_category->GetURL()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadWithImages(1, true), "is_bool", "Model: Shop_category->LoadWithImages()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadBySiteID(1, 1, true), "is_array", "Model: Shop_category->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->GetCategoriesForProductID(1), "is_array", "Model: Shop_category->GetCategoriesForProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadByProductID(1), "is_array", "Model: Shop_category->LoadByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadByParentID(1, 1), "is_array", "Model: Shop_category->LoadByParentID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadBySiteIDParentID(2, 1, 1), "is_array", "Model: Shop_category->LoadBySiteIDParentID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadByDatasourceCategoryID(1, 1), "is_bool", "Model: Shop_category->LoadByDatasourceCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->GetParentIDs(), "is_string", "Model: Shop_category->GetParentIDs()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->GetPath(false), "is_string", "Model: Shop_category->GetPath()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->GetBreadcrumbs(), "is_string", "Model: Shop_category->GetBreadcrumbs()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadAll(), "is_array", "Model: Shop_category->LoadAll()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadAllBare(), "is_array", "Model: Shop_category->LoadAllBare()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->get(), "is_array", "Model: Shop_category->get()", "Checking for correct data type.");
			$this->unit->run($this->Shop_category->LoadForRouteBuilder(1), "is_array", "Model: Shop_category->LoadForRouteBuilder()", "Checking for correct data type.");

			$this->unit->run(new Shop_config(), "is_object", "Model: Shop_config->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_config->load(1), "is_bool", "Model: Shop_config->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_config->LoadBySiteID(1), "is_array", "Model: Shop_config->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_config->LoadAssocBySiteID(1), "is_array", "Model: Shop_config->LoadAssocBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_config->get(), "is_array", "Model: Shop_config->get()", "Checking for correct data type.");

			$this->unit->run(new Shop_currency(), "is_object", "Model: Shop_currency->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_currency->load(1), "is_bool", "Model: Shop_currency->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_currency->LoadByCode("x"), "is_bool", "Model: Shop_currency->LoadByCode()", "Checking for correct data type.");
			$this->unit->run($this->Shop_currency->LoadAll(), "is_array", "Model: Shop_currency->LoadAll()", "Checking for correct data type.");

			$this->unit->run(new Shop_datasource(), "is_object", "Model: Shop_datasource->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_datasource->load(1), "is_bool", "Model: Shop_datasource->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_datasource->LoadBySiteID(1), "is_array", "Model: Shop_datasource->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_datasource->get(), "is_array", "Model: Shop_datasource->get()", "Checking for correct data type.");

			$this->unit->run(new Shop_image(), "is_object", "Model: Shop_image->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_image->load(1), "is_bool", "Model: Shop_image->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_image->LoadBySiteID(1), "is_array", "Model: Shop_image->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_image->LoadByProductID(1), "is_array", "Model: Shop_image->LoadByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_image->LoadByFilename("x"), "is_bool", "Model: Shop_image->LoadByFilename()", "Checking for correct data type.");

			$this->unit->run(new Shop_layout_specific_data(), "is_object", "Model: Shop_layout_specific_data->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_layout_specific_data->load(1), "is_bool", "Model: Shop_layout_specific_data->Load()", "Checking for correct data type.");

			$this->unit->run(new Shop_order_status(), "is_object", "Model: Shop_order_status->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order_status->load(1), "is_bool", "Model: Shop_order_status->Load()", "Checking for correct data type.");

			$this->unit->run(new Shop_order(), "is_object", "Model: Shop_order->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->load(1), "is_bool", "Model: Shop_order->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->add_tracking_number("x"), "is_null", "Model: Shop_order->add_tracking_number()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->LoadWithDetails(1), "is_bool", "Model: Shop_order->LoadWithDetails()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->LoadByUserID(1), "is_array", "Model: Shop_order->LoadByUserID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->update_status(1, 1), "is_bool", "Model: Shop_order->update_status()", "Checking for correct data type.");
			$this->unit->run($this->Shop_order->get_tracking_urls(), "is_array", "Model: Shop_order->get_tracking_urls()", "Checking for correct data type.");

			$this->unit->run(new Shop_payment_gateway(), "is_object", "Model: Shop_payment_gateway->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_payment_gateway->load(1), "is_bool", "Model: Shop_payment_gateway->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_payment_gateway->LoadBySiteID(1), "is_array", "Model: Shop_payment_gateway->LoadBySiteID()", "Checking for correct data type.");

			$this->unit->run(new Shop_postalcode(), "is_object", "Model: Shop_postalcode->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_postalcode->load(1), "is_bool", "Model: Shop_postalcode->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_postalcode->LoadByCode("x"), "is_bool", "Model: Shop_postalcode->LoadByCode()", "Checking for correct data type.");

			$this->unit->run(new Shop_price(), "is_object", "Model: Shop_price->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->load(1), "is_bool", "Model: Shop_price->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->LoadProductPrices(1), "is_array", "Model: Shop_price->LoadProductPrices()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->LoadByProductID(1, 1), "is_array", "Model: Shop_price->LoadByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->add_price(1, 1, 1), "is_bool", "Model: Shop_price->add_price()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->delete_price(1, 1), "is_bool", "Model: Shop_price->delete_price()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->delete_all(1), "is_null", "Model: Shop_price->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Shop_price->delete_others(1, "1,2"), "is_null", "Model: Shop_price->delete_others()", "Checking for correct data type.");

			$this->unit->run(new Shop_product_layout(), "is_object", "Model: Shop_product_layout->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_layout->load(1), "is_bool", "Model: Shop_product_layout->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_layout->get_layout_options(), "is_array", "Model: Shop_product_layout->get_layout_options()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_layout->get_layout_options_for_product(1), "is_array", "Model: Shop_product_layout->get_layout_options_for_product()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_layout->get_layout_options_for_product_display(1), "is_array", "Model: Shop_product_layout->get_layout_options_for_product_display()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_layout->save_product_option(1, "x", "y"), "is_null", "Model: Shop_product_layout->save_product_option()", "Checking for correct data type.");

			$this->unit->run(new Shop_product_option(), "is_object", "Model: Shop_product_option->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_option->load(1), "is_bool", "Model: Shop_product_option->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_option->LoadBySiteID(1), "is_array", "Model: Shop_product_option->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_option->LoadByProductOrCategoryID(), "is_array", "Model: Shop_product_option->LoadByProductOrCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_option->get(), "is_array", "Model: Shop_product_option->get()", "Checking for correct data type.");

			$this->unit->run(new Shop_product_related(), "is_object", "Model: Shop_product_related->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_related->load(1), "is_bool", "Model: Shop_product_related->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_related->LoadByProductID(1), "is_array", "Model: Shop_product_related->LoadByProductID()", "Checking for correct data type.");

			$this->unit->run(new Shop_product(), "is_object", "Model: Shop_product->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->load(1), "is_bool", "Model: Shop_product->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->GetURL(), "is_string", "Model: Shop_product->GetURL()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->GetBreadcrumbs(), "is_string", "Model: Shop_product->GetBreadcrumbs()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->GetCategoryPath(false), "is_string", "Model: Shop_product->GetCategoryPath()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadWithImages(1, true), "is_bool", "Model: Shop_product->LoadWithImages()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadForAdmin(1), "is_bool", "Model: Shop_product->LoadForAdmin()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadBySiteID(1), "is_array", "Model: Shop_product->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadByDatasourceProductID(1, 1), "is_bool", "Model: Shop_product->LoadByDatasourceProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadByDatasourceID(1), "is_array", "Model: Shop_product->LoadByDatasourceID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadFeedByCategoryID(1, 1), "is_array", "Model: Shop_product->LoadFeedByCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadByCategoryID(1, 1), "is_array", "Model: Shop_product->LoadByCategoryID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadProductsToUpdate(1, 1), "is_array", "Model: Shop_product->LoadProductsToUpdate()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadByQuery(true, 1), "is_array", "Model: Shop_product->LoadByQuery()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadLayoutSpecificDataValues(null, false), "is_array", "Model: Shop_product->LoadLayoutSpecificDataValues()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadAllPrices(1), "is_object", "Model: Shop_product->LoadAllPrices()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadAll(), "is_array", "Model: Shop_product->LoadAll()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->LoadAllBare(), "is_array", "Model: Shop_product->LoadAllBare()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->get(), "is_array", "Model: Shop_product->get()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->publish_products(1), "is_bool", "Model: Shop_product->publish_products()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->unpublish_products(1), "is_bool", "Model: Shop_product->unpublish_products()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product->update_field(1, "published", 1), "is_bool", "Model: Shop_product->update_field()", "Checking for correct data type.");

			$this->unit->run(new Shop_product_variant(), "is_object", "Model: Shop_product_variant->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_variant->load(1), "is_bool", "Model: Shop_product_variant->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_variant->LoadByProductID(1), "is_array", "Model: Shop_product_variant->LoadByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_variant->LoadAllByProductID(1), "is_array", "Model: Shop_product_variant->LoadAllByProductID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_variant->delete_all(1), "is_null", "Model: Shop_product_variant->delete_all()", "Checking for correct data type.");
			$this->unit->run($this->Shop_product_variant->delete_others(1, "1,2"), "is_null", "Model: Shop_product_variant->delete_others()", "Checking for correct data type.");

			$this->unit->run(new Shop_route(), "is_object", "Model: Shop_route->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->load(1), "is_bool", "Model: Shop_route->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->LoadAllBySiteID(1), "is_array", "Model: Shop_route->LoadAllBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->LoadBySiteID(1, "x"), "is_bool", "Model: Shop_route->LoadBySiteID()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->load_by_url_fragment(1, "x"), "is_array", "Model: Shop_route->load_by_url_fragment()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->build_product_route($p), "is_null", "Model: Shop_route->build_product_route()", "Checking for correct data type.");
			$this->unit->run($this->Shop_route->build_category_route($c), "is_null", "Model: Shop_route->build_category_route()", "Checking for correct data type.");

			$this->unit->run(new Shop_shipping_gateway(), "is_object", "Model: Shop_shipping_gateway->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_shipping_gateway->load(1), "is_bool", "Model: Shop_shipping_gateway->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_shipping_gateway->LoadBySiteID(1), "is_array", "Model: Shop_shipping_gateway->LoadBySiteID()", "Checking for correct data type.");

			$this->unit->run(new Shop_shopper_group(), "is_object", "Model: Shop_shopper_group->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_shopper_group->load(1), "is_bool", "Model: Shop_shopper_group->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_shopper_group->LoadBySiteID(1), "is_array", "Model: Shop_shopper_group->LoadBySiteID()", "Checking for correct data type.");

			$this->unit->run(new Shop_tax(), "is_object", "Model: Shop_tax->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_tax->load(1), "is_bool", "Model: Shop_tax->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_tax->LoadBySiteIDForProvince(1, "CA", "ON"), "is_array", "Model: Shop_tax->LoadBySiteIDForProvince()", "Checking for correct data type.");

			$this->unit->run(new Shop_vehicle(), "is_object", "Model: Shop_vehicle->__construct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->load(1), "is_bool", "Model: Shop_vehicle->Load()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetVehiclesForProduct(1), "is_array", "Model: Shop_vehicle->GetVehiclesForProduct()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetVehicles(), "is_array", "Model: Shop_vehicle->GetVehicles()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetYears(), "is_array", "Model: Shop_vehicle->GetYears()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetMakes(2016), "is_array", "Model: Shop_vehicle->GetMakes()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetModels(2016, "Toyota"), "is_array", "Model: Shop_vehicle->GetModels()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetStyles(2016, "Toyota", "Matrix"), "is_array", "Model: Shop_vehicle->GetStyles()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetProducts(2016, "Toyota", "Matrix", "x"), "is_array", "Model: Shop_vehicle->GetProducts()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetMakesMulti(2016), "is_array", "Model: Shop_vehicle->GetMakesMulti()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetModelsMulti(2016, "Toyota"), "is_array", "Model: Shop_vehicle->GetModelsMulti()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetStylesMulti(2016, "Toyota", "Matrix"), "is_array", "Model: Shop_vehicle->GetStylesMulti()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->GetProductsMulti(2016, "Toyota", "Matrix", "x"), "is_array", "Model: Shop_vehicle->GetProductsMulti()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->link_to_product(1, 1), "is_null", "Model: Shop_vehicle->link_to_product()", "Checking for correct data type.");
			$this->unit->run($this->Shop_vehicle->delete_link(1, 1), "is_null", "Model: Shop_vehicle->delete_link()", "Checking for correct data type.");

			// Create dummy page/domain objects.
			$page = new Page();
			$page->load(1);

			$domain = new Domain();
			$domain->load(1);

			// Load up every module, give it some fake settings, run its render() function, and see if there are any PHP errors.
			$query = $this->db->query("SELECT * FROM `modules`");

			foreach($query->result() as $module)
			{
				require_once(ABSOLUTE_PATH . "/modules/" . $module->file_name . "/" . $module->file_name . ".php");
				$m = new $module->class_name();

				$settings = file_get_contents(APPLICATION_PATH . "/views/modules/" . $module->file_name . "/settings.php");
				preg_match_all('/"(param_.*?)"/', $settings, $matches);
				$settings = array_unique($matches[1]);

				foreach($settings as $setting)
				{
					$setting 		= str_replace("param_", "", $setting);
					$value 			= new StdClass();
					$value->key 	= $setting;
					$value->value 	= "x";

					if(preg_match('/directory/', $setting))
					{
						$value->value = "images/gallery";
					}
					else if(preg_match('/width/', $setting))
					{
						$value->value = 400;
					}
					else if(preg_match('/height/', $setting))
					{
						$value->value = 300;
					}
					else if(preg_match('/(_id|per_row)/', $setting))
					{
						$value->value = 1;
					}
					else if(preg_match('/breakpoint/', $setting))
					{
						$value->value = 100;
					}

					$m->set($value->key, $value->value);
				}

				ob_start();
				$m->render();
				$content = ob_get_contents();
				ob_end_clean();

				if(preg_match('/Severity: Warning/', $content))
				{
					$this->unit->run(false, "is_true", "Module Render: " . $module->name, "PHP warnings were detected.");
				}
				else if(preg_match('/Severity: Notice/', $content))
				{
					$this->unit->run(false, "is_true", "Module Render: " . $module->name, "PHP notices were detected.");
				}
				else
				{
					$this->unit->run(true, "is_true", "Module Render: " . $module->name, "No PHP warnings or notices were detected.");
				}
			}

			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/search/x"), "is_true", "URL: /ajax/search", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_image_dirs_array"), "is_true", "URL: /ajax/get_image_dirs_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_video_dirs_array"), "is_true", "URL: /ajax/get_video_dirs_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_video_files_array"), "is_true", "URL: /ajax/get_video_files_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_roles_array"), "is_true", "URL: /ajax/get_roles_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_categories_array"), "is_true", "URL: /ajax/get_categories_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_tags_array"), "is_true", "URL: /ajax/get_tags_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_categories_array"), "is_true", "URL: /ajax/get_shop_categories_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_menus_array"), "is_true", "URL: /ajax/get_menus_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_pages_array"), "is_true", "URL: /ajax/get_pages_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get/page"), "is_true", "URL: /ajax/get", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_documents"), "is_true", "URL: /ajax/get_documents", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_404s"), "is_true", "URL: /ajax/get_404s", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_domains"), "is_true", "URL: /ajax/get_domains", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_sites"), "is_true", "URL: /ajax/get_sites", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_categories"), "is_true", "URL: /ajax/get_categories", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_pages"), "is_true", "URL: /ajax/get_pages", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_redirects"), "is_true", "URL: /ajax/get_redirects", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_modules"), "is_true", "URL: /ajax/get_modules", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_menus"), "is_true", "URL: /ajax/get_menus", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_roles"), "is_true", "URL: /ajax/get_roles", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_events"), "is_true", "URL: /ajax/get_events", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_event_forms"), "is_true", "URL: /ajax/get_event_forms", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_directory_categories"), "is_true", "URL: /ajax/get_directory_categories", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_directory_companies"), "is_true", "URL: /ajax/get_directory_companies", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_module_settings"), "is_true", "URL: /ajax/get_module_settings", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_page_settings"), "is_true", "URL: /ajax/get_page_settings", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_content_types"), "is_true", "URL: /ajax/get_content_types", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_content_types_array"), "is_true", "URL: /ajax/get_content_types_array", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_popular_image_size"), "is_true", "URL: /ajax/get_popular_image_size", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_categories"), "is_true", "URL: /ajax/get_shop_categories", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_products"), "is_true", "URL: /ajax/get_shop_products", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_orders"), "is_true", "URL: /ajax/get_shop_orders", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_layout_options"), "is_true", "URL: /ajax/get_shop_layout_options", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_layout_specific_data"), "is_true", "URL: /ajax/get_shop_layout_specific_data", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_vehicles"), "is_true", "URL: /ajax/get_shop_vehicles", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_shopper_groups"), "is_true", "URL: /ajax/get_shop_shopper_groups", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_payment_gateways"), "is_true", "URL: /ajax/get_shop_payment_gateways", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_shop_shipping_gateways"), "is_true", "URL: /ajax/get_shop_shipping_gateways", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_icons"), "is_true", "URL: /ajax/get_icons", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_tokens/tags"), "is_true", "URL: /ajax/get_tokens/tags", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_appointments_for_module"), "is_true", "URL: /ajax/get_appointments_for_module", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_events_for_module"), "is_true", "URL: /ajax/get_events_for_module", "Checking for JSON data.");
			$this->unit->run($this->qaCurl(LIVE_SITE . "/ajax/get_documents_array"), "is_true", "URL: /ajax/get_documents_array", "Checking for JSON data.");

			$report = $this->unit->result();
			$passed = 0;
			$failed = 0;
			$data = '<div class="row">';
			$data .= '<div class="col-12 col-md-6 bold">Test</div>';
			$data .= '<div class="col-12 col-md-6 bold">Notes</div>';

			foreach($report as $r)
			{
				if($r["Result"] == "Passed")
				{
					$passed++;
				}
				else if($r["Result"] == "Failed")
				{
					$failed++;
				}

				$data .= '<div class="col-12 col-md-6"><em class="fas fa-fw ' . ($r["Result"] == "Passed" ? "fa-check-circle green" : "fa-times-circle red") . '" aria-hidden="true"></em> '  . $r["Test Name"] . '</div>';
				$data .= '<div class="col-12 col-md-6">' . $r["Notes"] . '</div>';
			}
			$data .= '</div>';

			$this->data["content"] = '<h2>Unit Tests</h2>';
			$this->data["content"] .= '<h3><span class="green">' . $passed . ' Tests Passed</span> / <span class="red">' . $failed. ' Tests Failed</span></h3>';
			$this->data["content"] .= '<p>Scroll down to also ensure that none of these tests generated PHP warnings.</p>';
			$this->data["content"] .= '<div class="row">' . $data . '</div>';

			// Look at every model file's functions and see if we've missed any.
			$ignore_functions = array("__construct", "populate", "readPostVars", "load", "delete", "save", "get", "tokenize", "__get", "initialize", "start", "template", "modules", "finalize", "turn_cache_off", "end", "login_required", "admin_login_required", "bare_init", "http404", "fourOhFour", "clean_css_cache", "clean_application_cache", "denied", "get_instance");
			$models = scandir(APPLICATION_PATH . "/models");

			foreach($models as $model)
			{
				if($model == "." || $model == ".." || $model == "index.html" || is_dir(APPLICATION_PATH . "/models/" . $model))
				{
					continue;
				}
				$model = str_replace(".php", "", $model);
				$this->load->model($model);
				$m = new $model();
				$functions = get_class_methods($m);

				foreach($functions as $function)
				{
					if(in_array($function, $ignore_functions))
					{
						continue;
					}

					if(stripos($data, "Model: " . $model . "->" . $function) === false)
					{
						$this->errors[] = $model . "->" . $function . "() not in unit tests.";
					}
				}
			}

			$models = scandir(APPLICATION_PATH . "/models/shop");

			foreach($models as $model)
			{
				if($model == "." || $model == ".." || $model == "index.html" || is_dir(APPLICATION_PATH . "/models/shop/" . $model))
				{
					continue;
				}
				$model = str_replace(".php", "", $model);
				$this->load->model("shop/" . $model);
				$m = new $model();
				$functions = get_class_methods($m);

				foreach($functions as $function)
				{
					if(in_array($function, $ignore_functions))
					{
						continue;
					}

					if(stripos($data, "Model: " . $model . "->" . $function) === false)
					{
						$this->errors[] = $model . "->" . $function . "() not in unit tests.";
					}
				}
			}

			// Check if any AJAX calls are untested.
			require_once(APPLICATION_PATH . "/controllers/Ajax.php");
			$ajax = new Ajax();

			$functions = get_class_methods($ajax);

			foreach($functions as $function)
			{
				if(in_array($function, $ignore_functions))
				{
					continue;
				}

				if(stripos($data, "URL: /ajax/" . $function) === false)
				{
					$this->errors[] = "/ajax/" . $function . "() not in unit tests.";
				}
			}
		}
		else
		{
			$this->data["content"] = '<h2>Unit Tests</h2>';
			$this->errors[] = "You can only run unit testing when ENVIRONMENT is set to 'development' and it WILL WIPE SOME OF YOUR DATA.";
			$this->errors[] = "DO NOT RUN UNIT TESTING ON A LIVE WEBSITE!";
		}
		$this->load->view("common/header", $this->data);
		$this->load->view("common/message", array("messages" => $this->messages, "errors" => $this->errors));
		$this->load->view("index", $this->data);
		$this->load->view("common/footer", $this->data);
	}

	private function qaCurl($url)
	{
		$ch = curl_init($url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);

		try
		{
			json_decode(curl_exec($ch));
			return true;
		}
		catch(Exception $e)
		{
			return false;
		}

		return false;
	}

	public function test_mail_settings()
	{
		$settings 					= array();
		$settings["mail_to"] 		= $this->input->post("mail_to");
		$settings["mail_from"] 		= $this->input->post("mail_from");
		$settings["mail_replyto"] 	= $this->input->post("mail_replyto");
		$settings["mail_host"] 		= $this->input->post("mail_host");
		$settings["mail_port"] 		= $this->input->post("mail_port");
		$settings["mail_tls"] 		= $this->input->post("mail_tls");
		$settings["mail_username"] 	= $this->input->post("mail_username");
		$settings["mail_password"] 	= $this->input->post("mail_password");

		if($this->_test_mail_settings($settings))
		{
			http_response_code(200);
			die();
		}

		// Try a bunch of variations to the settings entered.
		// If using TLS on the wrong port, try the right port.
		if($settings["mail_tls"] == "tls" && $settings["mail_port"] != "587")
		{
			$settings["mail_port"] = "587";

			if($this->_test_mail_settings($settings))
			{
				http_response_code(200);
				die();
			}
		}
		// If using SSL on the wrong port, try the right port.
		if($settings["mail_tls"] == "ssl" && $settings["mail_port"] != "465")
		{
			$settings["mail_port"] = "465";

			if($this->_test_mail_settings($settings))
			{
				http_response_code(200);
				die();
			}
		}

		http_response_code(409);
		die("No working mail settings could be determined.");
	}

	private function _test_mail_settings($settings)
	{
		$mail = new PHPMailer();
		$mail->IsSMTP();
		$mail->IsHTML(true);
	    $mail->SMTPOptions = array
		(
		    'ssl' => array(
		        'verify_peer' => false,
		        'verify_peer_name' => false,
		        'allow_self_signed' => true
		    )
		);
		$mail->SMTPSecure		= $settings["mail_tls"];
	    $mail->SMTPAuth			= ($settings["mail_username"] && $settings["mail_password"]);
	    $mail->Timeout 			= 30;
	    $mail->SMTPKeepAlive 	= true;
		$mail->Host				= $settings["mail_host"];
		$mail->Port				= $settings["mail_port"];
		$mail->Username			= $settings["mail_username"];
		$mail->Password			= $settings["mail_password"];
		$mail->SetFrom($settings["mail_username"], $settings["mail_from"]);
	    $mail->ClearReplyTos();
		$mail->AddReplyTo($settings["mail_replyto"], $settings["mail_from"]);
		$mail->AddAddress($settings["mail_to"]);
		$mail->Subject = COMPANY_NAME . " Mail Settings Test";
		$mail->Body = "<p>The following mail settings worked:</p>";
		$mail->Body .= '<table width="100%" border="0">';
		$mail->Body .= '<tr><td>Server</td><td>' . $settings["mail_host"] . '</td></tr>';
		$mail->Body .= '<tr><td>Port</td><td>' . $settings["mail_port"] . '</td></tr>';
		$mail->Body .= '<tr><td>SSL/TLS</td><td>' . strtoupper($settings["mail_tls"]) . '</td></tr>';
		$mail->Body .= '<tr><td>Username</td><td>' . $settings["mail_username"] . '</td></tr>';
		$mail->Body .= '<tr><td>Password</td><td>[hidden]</td></tr>';
		$mail->Body .= '</table>';

		return $mail->Send();
	}

	public function environment($environment)
	{
		$content = file_get_contents(ABSOLUTE_PATH . "/index.php");
		$content = preg_replace('/define\(\'ENVIRONMENT\', \'.*\'\);/', "define('ENVIRONMENT', '" . $environment . "');", $content);

		file_put_contents(ABSOLUTE_PATH . "/index.php", $content);

		$this->messages[] = "Your website's environment was changed to " . $environment . ".";
		$this->index();
	}

	public function optimize_images()
	{
		$this->init();
		ini_set('gd.jpeg_ignore_warning', true);

		$obliterated = false;

		if(file_exists(ABSOLUTE_PATH . "/images/thumbnails"))
		{
			$this->rrmdir(ABSOLUTE_PATH . "/images/thumbnails");
			mkdir(ABSOLUTE_PATH . "/images/thumbnails");
			$obliterated = true;
		}
		else
		{
			mkdir(ABSOLUTE_PATH . "/images/thumbnails");
		}
		touch(ABSOLUTE_PATH . "/images/thumbnails/index.html");

		$bytes_saved = 0;
		$bytes_saved += $this->_optimize_images(ABSOLUTE_PATH . '/images');
		$bytes_saved += $this->_optimize_images(ABSOLUTE_PATH . '/templates');

		// Pop the total summary to the top of the messages.
		array_unshift($this->messages, "Saved a total of " . number_format($bytes_saved / 1024, 1) . " KB");

		if($obliterated)
		{
			array_unshift($this->messages, "/images/thumbnails was emptied.");
		}

		$this->index();
	}

	private function rrmdir($src)
	{
		$dir = opendir($src);
		while(false !== ( $file = readdir($dir)) )
		{
			if (( $file != '.' ) && ( $file != '..' ))
			{
				$full = $src . '/' . $file;
				if ( is_dir($full) )
				{
					$this->rrmdir($full);
				}
				else
				{
					unlink($full);
				}
			}
		}
		closedir($dir);
		rmdir($src);
	}

	private function _optimize_images($path)
	{
		$bytes_saved = 0;

		// We must use opendir rather than scandir() or glob() in order to handle massive directories.
		if($handle = opendir($path))
		{
		    while(false !== ($image = readdir($handle)))
		    {
		        if($image == "." || $image == "..")
		        {
		        	continue;
		        }

		        set_time_limit(20);

				if(is_dir($path . "/" . $image))
				{
					$bytes_saved += $this->_optimize_images($path . "/" . $image);
				}

				try
				{
					$size = @getimagesize($path . "/" . $image);

					if(preg_match('/jpe?g/i', $size["mime"]))
					{
						$bytes_saved += $this->_optimize_jpg($path . "/" . $image);
					}
					else if(preg_match('/png/i', $size["mime"]))
					{
						$bytes_saved += $this->_optimize_png($path . "/" . $image);
					}
				}
				catch(Exception $e)
				{
					$this->errors[] = str_replace(ABSOLUTE_PATH, "", $path . "/" . $image) . " failed to be optimized.";
					return 0;
				}
		    }

		    closedir($handle);
		}

		return $bytes_saved;
	}

	private function _optimize_png($image)
	{
		try
		{
			$size1 	= filesize($image);
			$i 		= imagecreatefrompng($image);

			// As per Google's suggestion, images over 10KB should be progressive.
			// Not sure if this is applicable to PNG but they suggest this for JPG so why not eh?
			if($size1 > 10240)
			{
				imageinterlace($i);
			}

			imagepng($i, $image, 9, PNG_NO_FILTER);
			$size2 = filesize($image);

			if($size1-$size2 > 0)
			{
				$this->messages[] = str_replace(ABSOLUTE_PATH, "", $image) . " reduced by " . number_format(($size1-$size2) / 1024, 1) . " KB";
				@imagedestroy($i);
				return ($size1-$size2);
			}
			// else if($size2-$size1 > 0)
			// {
			// 	devecho("Gained " . ($size1-$size2) . " bytes");
			// }

			@imagedestroy($i);
		}
		catch(Exception $e)
		{
			$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " failed to be optimized.";
			@imagedestroy($i);
			return 0;
		}

		@imagedestroy($i);
		return 0;
	}

	private function _optimize_jpg($image)
	{
		// Check for JPG file header and footer - also try to fix it
		if( false !== (@$fd = fopen($image, 'r+b' )) )
		{
			if ( fread($fd,2)==chr(255).chr(216) )
			{
				fseek ( $fd, -2, SEEK_END );
				if ( fread($fd,2)==chr(255).chr(217) )
				{
					fclose($fd);
					// header/footer is ok
				}
				else
				{
					if (!fwrite($fd,chr(255).chr(217)) )
					{
						// header/footer could not be fixed
						fclose($fd);
						$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " has an invalid JPG header or footer and could not be fixed.";
						return 0;
					}
					fclose($fd);
				}
			}
			else
			{
				fclose($fd);
				$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " could not be read.";
				return 0;
			}
		}
		else
		{
			$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " could not be opened.";
			return 0;
		}

		try
		{
			$temp 	= $image . ".optimized";
			$size1 	= filesize($image);
			$i 		= imagecreatefromjpeg($image);

			// As per Google's suggestion, images over 10KB should be progressive.
			if($size1 > 10240)
			{
				imageinterlace($i);
			}

			// Starting with quality 100, continually compress the original image.
			// Compressing an already compressed image with a higher compression yields a larger file.
			// When we find a JPG quality that creates a smaller file, we can assume that's the approximate existing compression.
			// If we find an image that is compressed approximately higher than 85%, reduce to 85%.
			for($q=100; $q>=86; $q--)
			{
				try
				{
					@unlink($temp);
					imagejpeg($i, $temp, $q);
					$size2 = filesize($temp);

					if($size1-$size2 > 0)
					{
						@unlink($temp);
						imagejpeg($i, $image, 85);
						$size2 = filesize($image);
						$this->messages[] = str_replace(ABSOLUTE_PATH, "", $image) . " reduced by " . number_format(($size1-$size2) / 1024, 1) . " KB";

						@unlink($temp);
						@imagedestroy($i);

						return ($size1-$size2);
					}
				}
				catch(Exception $e)
				{
					$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " failed to be optimized.";
					break;
				}
			}
			@unlink($temp);
			@imagedestroy($i);
		}
		catch(Exception $e)
		{
			$this->errors[] = str_replace(ABSOLUTE_PATH, "", $image) . " failed to be optimized.";
			@unlink($temp);
			@imagedestroy($i);
			return 0;
		}

		@unlink($temp);
		@imagedestroy($i);
		return 0;
	}

	public function moduleBench()
	{
		$this->output->enable_profiler();
		$this->admin_login_required();
		$query = $this->db->query("SELECT * FROM `modules`");

		foreach($query->result() as $module)
		{
			$start = microtime();

			require_once(ABSOLUTE_PATH . "/modules/" . $module->file_name . "/" . $module->file_name . ".php");
			$m = new $module->class_name();

			$settings = file_get_contents(APPLICATION_PATH . "/views/modules/" . $module->file_name . "/settings.php");
			preg_match_all('/"(param_.*?)"/', $settings, $matches);
			$settings = array_unique($matches[1]);

			foreach($settings as $setting)
			{
				$setting 		= str_replace("param_", "", $setting);
				$value 			= new StdClass();
				$value->key 	= $setting;
				$value->value 	= "x";

				if(preg_match('/directory/', $setting))
				{
					$value->value = "images/gallery";
				}
				else if(preg_match('/width/', $setting))
				{
					$value->value = 400;
				}
				else if(preg_match('/height/', $setting))
				{
					$value->value = 300;
				}
				else if(preg_match('/(_id|per_row)/', $setting))
				{
					$value->value = 1;
				}
				else if(preg_match('/breakpoint/', $setting))
				{
					$value->value = 100;
				}

				$m->set($value->key, $value->value);
			}

			ob_start();
			$m->render();
			$content = ob_get_contents();
			ob_end_clean();

			$end = microtime();

			devecho($module->file_name . " took " . number_format($end-$start, 5));
		}
	}

	public function sitecopy()
	{
		$this->load->model(array(
"Page",
"Content_value",
"Acl",
"Moduleinstance",
"Modulesetting",
"Menu",
"Menuitem",
"Document"));

		$from_site_id = 2;
		$site_ids     = array(4,5,6,7,8,9,11,12,13,14,15);
		$page_ids     = array();
		$module_ids   = array();

		foreach($site_ids as $site_id)
		{
			$q = $this->db->query("SELECT * FROM pages WHERE site_id=?", array($from_site_id));

			foreach($q->result() as $row)
			{
				if($row->page_id == 30) // Development milestones - skip
				{
					continue;
				}

				$p = new Page();
				$p->populate($row);
				$p->page_id = null;
				$p->site_id = $site_id;
				$p->save();

				$page_ids[$row->page_id] = $p->page_id;

				$values = $this->content_value->LoadByPageID($row->page_id);

				foreach($values as $value)
				{
					$v = new Content_value();
					$v->populate($value);
					$v->content_value_id = null;
					$v->page_id = $p->page_id;
					$v->save();
				}

				$acls = $this->acl->loadByPageID($row->page_id);

				foreach($acls as $acl)
				{
					$a = new Acl();
					$a->populate($acl);
					$a->acl_id = null;
					$a->page_id = $p->page_id;
					$a->save();
				}
			}

			$q = $this->db->query("SELECT * FROM module_instances WHERE site_id=?", array($from_site_id));

			foreach($q->result() as $row)
			{
				$mi = new Moduleinstance();
				$mi->populate($row);
				$mi->module_instance_id = null;
				$mi->site_id = $site_id;
				$mi->save();

				$module_ids[$row->module_instance_id] = $mi->module_instance_id;

				$settings = $this->modulesetting->load_by_instance_id($row->module_instance_id);

				foreach($settings as $setting)
				{
					$s = new Modulesetting();
					$s->populate($setting);
					$s->module_setting_id = null;
					$s->module_instance_id = $mi->module_instance_id;
					$s->save();
				}

				$acls = $this->acl->loadByModuleInstanceID($row->module_id);

				foreach($acls as $acl)
				{
					$a = new Acl();
					$a->populate($acl);
					$a->acl_id = null;
					$a->module_instance_id = $mi->module_instance_id;
					$a->save();
				}
			}

			$q = $this->db->query("SELECT * FROM menus WHERE site_id=?", array($from_site_id));

			foreach($q->result() as $row)
			{
				$m = new Menu();
				$m->populate($row);
				$m->menu_id = null;
				$m->site_id = $site_id;
				$m->save();

				$q2 = $this->db->query("SELECT * FROM menu_items WHERE menu_id=?", array($row->menu_id));

				foreach($q2->result() as $row2)
				{
					$mi = new Menuitem();
					$mi->populate($row2);
					$mi->menu_item_id = null;
					$mi->parent_id = null;
					$mi->menu_id = $m->menu_id;

					if($mi->page_id)
					{
						$mi->page_id = $page_ids[$mi->page_id];
					}
					$mi->save();

					$acls = $this->acl->loadByMenuItemID($row2->menu_item_id);

					foreach($acls as $acl)
					{
						$a = new Acl();
						$a->populate($acl);
						$a->acl_id = null;
						$a->menu_item_id = $mi->menu_item_id;
						$a->save();
					}
				}
			}

			// $q = $this->db->query("SELECT * FROM documents WHERE site_id=?", array($from_site_id));

			// foreach($q->result() as $row)
			// {
			// 	$d = new Document();
			// 	$d->populate($row);
			// 	$d->document_id = null;
			// 	$d->site_id = $site_id;
			// 	$d->save();

			// 	$acls = $this->acl->loadByDocumentID($row->document_id);

			// 	foreach($acls as $acl)
			// 	{
			// 		$a = new Acl();
			// 		$a->populate($acl);
			// 		$a->acl_id = null;
			// 		$a->document_id = $d->document_id;
			// 		$a->save();
			// 	}
			// }
		}

		foreach($page_ids as $page_id=>$new_page_id)
		{
			foreach($module_ids as $module_id=>$new_module_id)
			{
				$q = $this->db->query("SELECT * FROM modules_pages WHERE module_instance_id=? AND page_id=?",
				                      array($module_id, $page_id));

				foreach($q->result() as $row)
				{
					$this->db->query("INSERT INTO modules_pages SET module_instance_id=?, page_id=?",
					                 array($new_module_id, $new_page_id));
				}
			}
		}
	}

	public function cleanup()
	{
		$this->init();
		$this->clean_css_cache();
		$this->clean_application_cache();

		$this->messages[] = "CSS cache was cleaned.";
		$this->messages[] = "Application cache was cleaned.";

		$dir_iterator = new RecursiveDirectoryIterator(ABSOLUTE_PATH . "/images");
		$iterator     = new RecursiveIteratorIterator($dir_iterator, RecursiveIteratorIterator::SELF_FIRST);
		$files        = array();
		$bytes        = 0;
		$unused       = 0;

		foreach($iterator as $file)
		{
			$file = (string)$file;

			// Ignore jQuery UI theme files.
			if(preg_match('/\/ui-.*/', $file))
			{
				continue;
			}

			// Ignore shop images. They have their own cleanup function.
			if(preg_match('/\/shop\/.*/', $file))
			{
				continue;
			}

			// Ignore thumbnails.
			if(preg_match('/\/thumbnails\/.*/', $file))
			{
				continue;
			}

			// Ignore @2x, we'll clean those up if their main image is unused.
			if(preg_match('/@2x\.(jpe?g|png|gif|webp)$/', $file))
			{
				continue;
			}

			if(is_file($file) && !preg_match('/\.html$/', $file))
		    {
		    	$files[] = str_replace(ABSOLUTE_PATH . "/", "", $file);
		    }
		}

		foreach($files as $file)
		{
			$used = false;

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `content_values` WHERE `value` LIKE ?", array("%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `module_settings` WHERE `value` LIKE ?", array("%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `pages` WHERE `content` LIKE ?", array("%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `shop_products` WHERE `product_short_description` LIKE ? OR `product_description` LIKE ?", array("%" . $file . "%", "%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `shop_categories` WHERE `category_description` LIKE ?", array("%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			if(!$used)
			{
				$q = $this->db->query("SELECT * FROM `document` WHERE `description` LIKE ?", array("%" . $file . "%"));

				if($q->num_rows())
				{
					$used = true;
				}
			}

			$dir = $file;
			$failsafe = 0;

			while(strpos($dir, "/") !== false)
			{
				$failsafe++;
				$dir = dirname($dir);

				if(!$used)
				{
					$q = $this->db->query("SELECT * FROM `content_values` WHERE `value` = ? OR `value` = ?", array("/" . $dir, $dir));

					if($q->num_rows())
					{
						$used = true;
					}
				}

				if(!$used)
				{
					$q = $this->db->query("SELECT * FROM `module_settings` WHERE `value` = ? OR `value` = ?", array("/" . $dir, $dir));

					if($q->num_rows())
					{
						$used = true;
					}
				}

				if($failsafe == 100)
				{
					break;
				}
			}

			if(!$used)
			{
				$unused++;
				$bytes += filesize(ABSOLUTE_PATH . "/" . $file);
				unlink(ABSOLUTE_PATH . "/" . $file);
				$retina = preg_replace('/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/', '@2x.$1', $file);

				if(file_exists($retina))
				{
					$unused++;
					$bytes += filesize(ABSOLUTE_PATH . "/" . $retina);
					unlink(ABSOLUTE_PATH . "/" . $retina);
				}

				$thumbs = glob(ABSOLUTE_PATH . "/" . preg_replace(array('/^images\//', '/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/'),
				                                                   array('images/thumbnails/', '_*x*.$1'), $file));
				$thumbs = array_merge($thumbs, glob(ABSOLUTE_PATH . "/" . preg_replace(array('/^images\//', '/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/'),
				                                                   array('images/thumbnails/', '_*x*@2x.$1'), $file)));
				$thumbs = array_merge($thumbs, glob(ABSOLUTE_PATH . "/" . preg_replace(array('/^images\//', '/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/'),
				                                                   array('images/thumbnails/', '_*x*_*_*.$1'), $file)));
				$thumbs = array_merge($thumbs, glob(ABSOLUTE_PATH . "/" . preg_replace(array('/^images\//', '/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/'),
				                                                   array('images/thumbnails/', '_*x*_*_*@2x.$1'), $file)));

				foreach($thumbs as $thumb)
				{
					$unused++;
					$bytes += filesize($thumb);
					unlink($thumb);
				}
			}
		}

		// Look for unused thumbnails.
		$dir_iterator = new RecursiveDirectoryIterator(ABSOLUTE_PATH . "/images/thumbnails");
		$iterator     = new RecursiveIteratorIterator($dir_iterator, RecursiveIteratorIterator::SELF_FIRST);
		$files        = array();

		foreach($iterator as $file)
		{
			$file = (string)$file;

			if(is_file($file) && preg_match('/\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/', $file))
		    {
		    	$files[] = str_replace(ABSOLUTE_PATH . "/", "", $file);
		    }
		}

		foreach($files as $file)
		{
			$original = ABSOLUTE_PATH . "/" . preg_replace(array('/images\/thumbnails\//', '/_[0-9]+x[0-9]+(_[0-9]+_[0-9]+)?(_[a-z]+_[a-z]+)?(@2x)?\.(jpe?g|png|gif|webp|JPE?G|PNG|GIF|WEBP)$/'),
			                         array('images/', '.$3'), $file);

			if(!file_exists($original))
			{
				$unused++;
				$bytes += filesize(ABSOLUTE_PATH . "/" . $file);
				unlink(ABSOLUTE_PATH . "/" . $file);
			}
		}

		$this->messages[] = number_format($unused, 0) . " unused images were deleted, saving " . bytes_format($bytes) . ".";
		$this->index();
	}
}
